<?php

namespace Rvvup\Payments\Component\Admin;

use Exception;
use Rvvup\Payments\Sdk\Exceptions\ApiError;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Traits\GatewayTrait;
use Rvvup\Payments\Traits\MetaDataTrait;
use WC_Admin_Meta_Boxes;
use WC_Product_Variation;
use WC_Product;

defined("ABSPATH") || exit();

class OrderTrackingMetaBoxComponent
{
    use MetaDataTrait;
    use GatewayTrait;

    private function __construct()
    {
        add_action("add_meta_boxes", [$this, "register"], 10, 2);
        add_action("woocommerce_process_shop_order_meta", [$this, "save"]);
    }

    public static function init()
    {
        new self();
    }

    /**
     * Register the meta box for Rvvup orders
     */
    public function register($type, $post_or_order)
    {
        $order = $post_or_order instanceof WC_Order ? $post_or_order : wc_get_order($post_or_order);
        if (!$order) {
            return;
        }
        if (strpos($order->get_payment_method(), "rvvup_gateway_") !== 0) {
            return;
        }
        add_meta_box(
            "rvvup-tracking-meta-box",
            __("Shipment Tracking", "rvvup-for-woocommerce"),
            [$this, "render"],
            wc_get_page_screen_id("shop-order"),
            "side",
            "high"
        );
    }

    /**
     * Render the meta box content.
     */
    public function render()
    {
        wp_nonce_field("rvvup_tracking_nonce", "rvvup_tracking_nonce_field"); ?>
        <p>
            <label for="rvvup_carrier_code"><?php esc_html_e("Carrier Code", "rvvup-for-woocommerce"); ?></label><br>
            <input type="text" name="rvvup_carrier_code" id="rvvup_carrier_code" style="width:100%;">
        </p>
        <p>
            <label for="rvvup_tracking_number"><?php esc_html_e(
                "Tracking Number",
                "rvvup-for-woocommerce"
            ); ?></label><br>
            <input type="text" name="rvvup_tracking_number" id="rvvup_tracking_number" style="width:100%;">
        </p>
        <p>
            <input type="submit" class="button button-primary" name="rvvup_tracking_submit"
                   value="<?php esc_attr_e("Add Tracking", "rvvup-for-woocommerce"); ?>">
        </p>
        <?php
    }

    /**
     * Handle saving
     * @throws \Exception
     */
    public function save($post_id)
    {
        if (
            !isset($_POST["rvvup_tracking_nonce_field"]) ||
            !wp_verify_nonce($_POST["rvvup_tracking_nonce_field"], "rvvup_tracking_nonce")
        ) {
            return;
        }

        $order = wc_get_order($post_id);
        if (!$order) {
            return;
        }
        if (strpos($order->get_payment_method(), "rvvup_gateway_") !== 0) {
            return;
        }

        if (!isset($_POST["rvvup_tracking_submit"])) {
            return;
        }
        $carrier_code = $_POST["rvvup_carrier_code"] ?? "";
        $tracking_number = $_POST["rvvup_tracking_number"] ?? "";
        if (empty($carrier_code)) {
            WC_Admin_Meta_Boxes::add_error(__("Shipment Tracking carrier code is required", "rvvup-for-woocommerce"));
            return;
        }

        if (empty($tracking_number)) {
            WC_Admin_Meta_Boxes::add_error(__("Shipment Tracking number is required", "rvvup-for-woocommerce"));
            return;
        }

        $request = [
            "trackingDetail" => [
                [
                    "trackingNumber" => $tracking_number,
                    "carrierCode" => "$carrier_code",
                    "title" => $carrier_code,
                ],
            ],
            "items" => [],
        ];
        $orderItems = $order->get_items();
        foreach ($orderItems as $item) {
            $productId = $item["product_id"];
            $productVariationId = $item["variation_id"];

            // Check if product has variation.
            if ($productVariationId) {
                $singleProductData = new WC_Product_Variation($productVariationId);
            } else {
                $singleProductData = new WC_Product($productId);
            }

            $productRequest = [
                "name" => $singleProductData->get_name(),
                "sku" => $singleProductData->get_sku(),
                "quantity" => $item["qty"],
            ];
            $request[] = $productRequest;
        }
        try {
            SdkProxy::createShipmentTracking($this->getMetaData($order->get_id(), "rvvup_order_id"), $request);
            $order->add_order_note(
                sprintf(
                    __(
                        "Shipment Tracking has been submitted. Carrier: %s, Tracking Number: %s",
                        "rvvup-for-woocommerce"
                    ),
                    $carrier_code,
                    $tracking_number
                )
            );
        } catch (ApiError $apiError) {
            WC_Admin_Meta_Boxes::add_error(
                sprintf(
                    __("Shipment Tracking error [%s]: %s", "rvvup-for-woocommerce"),
                    $apiError->getErrorCode(),
                    $apiError->getMessage()
                )
            );
            return;
        } catch (Exception $e) {
            WC_Admin_Meta_Boxes::add_error(
                sprintf(__("Shipment Tracking error: %s", "rvvup-for-woocommerce"), $e->getMessage())
            );
            return;
        }
    }
}
