<?php declare(strict_types=1);

namespace Rvvup\Payments\Controller;

use Rvvup\Payments\Sdk\Exceptions\ApiError;
use Rvvup\Payments\Service\GatewaySettingsManager;
use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Traits\EndpointTrait;

class CardSubmitCallbackEndpoint
{
    use EndpointTrait;

    public const ENDPOINT = "rvvup-card-submit-callback";

    public static function execute()
    {
        $nonce_value = wc_get_var($_REQUEST["nonce"], wc_get_var($_REQUEST["_wpnonce"], ""));
        $logger = LoggerManager::getInstance();
        $metadata = [];
        $orderId = sanitize_text_field($_POST["order_id"]);

        if (!wp_verify_nonce($nonce_value, self::ENDPOINT)) {
            $logger->error("Card submit callback failed with invalid nonce", [
                "nonce_value" => $nonce_value,
                "order_id" => $orderId,
            ]);
            self::sendGenericError();
        }

        $metadata["order_id"] = $orderId;
        $wcOrder = wc_get_order($orderId);
        if (!$wcOrder) {
            $logger->error("Card submit callback failed with woocommerce order not existing with id", $metadata);
            self::sendGenericError();
        }

        $rvvupOrderId = $wcOrder->get_meta("rvvup_order_id", true);
        $rvvupPaymentId = $wcOrder->get_meta("rvvup_payment_id", true);
        $metadata["rvvup_order_id"] = $rvvupOrderId;
        $metadata["rvvup_payment_id"] = $rvvupPaymentId;

        if (empty($rvvupOrderId) || empty($rvvupPaymentId)) {
            $logger->error("Card submit callback failed with empty rvvup payment data", $metadata);
            self::sendGenericError();
        }

        if (!isset($_POST["auth"])) {
            $logger->error("Card submit callback failed with authorization not set", $metadata);
            self::sendGenericError();
        }
        $authorizationResponse = sanitize_text_field($_POST["auth"]);
        if (empty($authorizationResponse)) {
            $logger->error("Card submit callback failed with authorization empty", $metadata);
            self::sendGenericError();
        }

        if (isset($_POST["three_d"])) {
            $threeDSecureResponse = sanitize_text_field($_POST["three_d"]);
        } else {
            $threeDSecureResponse = null;
        }

        try {
            if (GatewaySettingsManager::getInstance()->isDebugEnabled()) {
                LoggerManager::getInstance()->debug("Confirm Card Authorization called", $metadata);
            }
            SdkProxy::confirmCardAuthorization(
                $rvvupPaymentId,
                $rvvupOrderId,
                $authorizationResponse,
                $threeDSecureResponse
            );
            if (GatewaySettingsManager::getInstance()->isDebugEnabled()) {
                LoggerManager::getInstance()->debug("Successfully submitted card authorization", $metadata);
            }
        } catch (\Exception $e) {
            if ($e instanceof ApiError) {
                if ($e->getErrorCode() == "card_authorization_not_found") {
                    $logger->warning(
                        "Card authorization has not been processed fully, setting to retryable to check again",
                        $metadata
                    );
                    wp_send_json(["result" => "failure", "retryable" => true]);
                }
            }
            $message = $e->getMessage();
            $logger->error("Card submit callback failed to confirm auth with exception: " . $message, $metadata);
            wp_send_json(["result" => "failure", "retryable" => false, "message" => $message]);
        }

        wp_send_json(["result" => "success"]);
    }

    private static function sendGenericError()
    {
        wp_send_json([
            "result" => "failure",
            "retryable" => false,
            "message" => "Something went wrong, please try again.",
        ]);
    }
}
