<?php

declare(strict_types=1);

namespace Rvvup\Payments\Controller;

use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\SessionManager;
use Rvvup\Payments\Traits\EndpointTrait;

class ExpressDeleteEndpoint
{
    use EndpointTrait;

    public const ENDPOINT = "rvvup-express-delete";

    /**
     * @return void
     */
    public static function execute()
    {
        $logger = LoggerManager::getInstance();
        // Only accept JSON requests.
        if (!wp_is_json_request()) {
            wp_send_json_error("Cannot process this request");
        }

        $nonce_value = wc_get_var($_REQUEST["nonce"], wc_get_var($_REQUEST["_wpnonce"], ""));

        if (!wp_verify_nonce($nonce_value, self::ENDPOINT)) {
            wp_send_json_error("Cannot process this request");
        }

        $data = json_decode(file_get_contents("php://input"), true);

        if ($data === null || $data === false) {
            $logger->error("Failed to decode express payment data with message: " . json_last_error());

            self::addNoticeAndFail("Cannot process this request");

            return;
        }

        $rvvupExpressOrder = SessionManager::getInstance()->getRvvupExpressOrder();

        // If no session express order or request does not have an id, return with error.
        if (!is_array($rvvupExpressOrder) || !is_array($data) || !isset($rvvupExpressOrder["id"], $data["id"])) {
            $message = "No session express order or request does not have an id";
            $logger->error("Failed to unset Rvvup Express Order with: " . $message);
            self::addNoticeAndFail("Cannot process this request");
            return;
        }

        // If Session & Request rvvup express payment id do not match, return with error.
        if (sanitize_text_field($data["id"]) !== $rvvupExpressOrder["id"]) {
            $logger->error("Failed to unset Rvvup Express Order with: data missing.");
            self::addNoticeAndFail("Cannot process this request");
            return;
        }

        SessionManager::getInstance()->unsetRvvupExpressOrder();

        wp_send_json(["result" => "success"]);
    }
}
