<?php declare(strict_types=1);

namespace Rvvup\Payments\Controller;

use Exception;
use Rvvup\Payments\Service\GatewayLoader;
use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Traits\EndpointTrait;

class ExpressEndpoint
{
    use EndpointTrait;

    public const ENDPOINT = "rvvup-express";

    public static function execute()
    {
        $nonce_value = wc_get_var($_REQUEST["nonce"], wc_get_var($_REQUEST["_wpnonce"], ""));
        $logger = LoggerManager::getInstance();

        if (!wp_verify_nonce($nonce_value, "rvvup-express")) {
            $logger->error("Express payment failed with incorrect nonce value");
            self::addNoticeAndFail("Cannot process this request");
        }

        $data = json_decode(file_get_contents("php://input"), false);

        if ($data === null || $data === false) {
            $logger->error("Failed to decode express payment data with message: " . json_last_error());
            self::addNoticeAndFail("There was an error when processing your request");
            return;
        }

        $paymentMethod = sanitize_text_field($data->payment_method);

        if (stripos($paymentMethod, "rvvup_gateway_") !== 0) {
            $logger->error("Express payment failed with method not being rvvup");
            self::addNoticeAndFail("Selected payment method is invalid");
            return;
        }

        new GatewayLoader();

        $available_gateways = WC()->payment_gateways->get_available_payment_gateways();

        if (!isset($available_gateways[$paymentMethod])) {
            $logger->error("Express payment failed with current method is not available");
            self::addNoticeAndFail("Selected payment method is not available right now");

            return;
        }

        WC()->shipping->reset_shipping();
        WC()->cart->empty_cart(false);

        foreach ($data->products as $product) {
            $productId = absint($product->id);
            $quantity = absint($product->quantity);
            $variationId = absint($product->variation_id); // null = 0 for variationId in woocommerce

            $variation = !empty($product->variation) ? ((array) $product->variation) : [];

            $wcProduct = wc_get_product($productId);

            if (!$wcProduct) {
                self::addNoticeAndFail("Selected product is not available right now");

                return;
            }

            if ($variationId === 0 && $wcProduct->is_type("variable")) {
                $logger->error("Express payment failed with variation missing");
                self::addNoticeAndFail("Please select a variation to continue");
            }

            $metadata = [];
            $metadata["product_id"] = $productId;
            $metadata["quantity"] = $quantity;
            $metadata["variation_id"] = $variationId;
            $metadata["variation"] = $variation;

            try {
                if (!WC()->cart->add_to_cart($productId, $quantity, $variationId, $variation)) {
                    $logger->error(
                        "Failed to add product to cart on express payment with the following data.",
                        $metadata
                    );

                    self::addNoticeAndFail("There was an error when processing your request");

                    return;
                }
            } catch (Exception $ex) {
                $logger->error(
                    "Failed to add product to cart on express payment with message: " . $ex->getMessage(),
                    $metadata
                );

                self::addNoticeAndFail($ex->getMessage());

                return;
            }

            // Need to calculate totals as they handle discounts etc.
            WC()->cart->calculate_totals();
        }

        if (count(WC()->cart->get_cart_contents()) === 0) {
            $logger->error("Express payment failed with no items being present");
            self::addNoticeAndFail("No items have been selected");
            return;
        }

        try {
            // Should be an instance of \Rvvup\Payments\Gateway\Dynamic
            $result = $available_gateways[$paymentMethod]->process_cart_payment(WC()->cart);
        } catch (Exception $ex) {
            $logger->error("Express payment failed with error of gateway not being available");
            self::addNoticeAndFail("There was an error when processing your request");
            return;
        }

        if (isset($result["result"])) {
            wp_send_json($result);
        }
        $logger->error("Express payment failed with incorrect result from API");
        self::addNoticeAndFail("There was an error when processing your request");
    }
}
