<?php

declare(strict_types=1);

namespace Rvvup\Payments\Controller;

use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\SessionManager;
use Rvvup\Payments\Traits\EndpointTrait;

class ExpressUpdateEndpoint
{
    use EndpointTrait;

    public const ENDPOINT = "rvvup-express-update";

    /**
     * @return void
     */
    public static function execute()
    {
        // Only accept JSON requests.
        if (!wp_is_json_request()) {
            wp_send_json_error("Cannot process this request");
        }

        $nonce_value = wc_get_var($_REQUEST["nonce"], wc_get_var($_REQUEST["_wpnonce"], ""));

        if (!wp_verify_nonce($nonce_value, self::ENDPOINT)) {
            wp_send_json_error("Cannot process this request");
        }

        $data = json_decode(file_get_contents("php://input"), true);

        if ($data === null || $data === false) {
            LoggerManager::getInstance()->error(
                "Failed to decode express payment data with message: " . json_last_error()
            );

            self::addNoticeAndFail("Cannot process this request");

            return;
        }

        $rvvupExpressOrder = SessionManager::getInstance()->getRvvupExpressOrder();

        // If no session express order or request does not have an id, return with error.
        if (!is_array($rvvupExpressOrder) || !is_array($data) || !isset($rvvupExpressOrder["id"], $data["id"])) {
            self::addNoticeAndFail("Cannot process this request");

            return;
        }

        // If Session & Request rvvup express payment id do not match, return with error.
        if (sanitize_text_field($data["id"]) !== $rvvupExpressOrder["id"]) {
            self::addNoticeAndFail("Cannot process this request");

            return;
        }

        self::updateRvvupExpressSessionData($rvvupExpressOrder, $data);

        wp_send_json(["result" => "success"]);
    }

    /**
     * Update the Rvvup Express session data.
     *
     * Keep the original session data (id, payment_method_code) & add billing & shipping address.
     *
     * @param array $rvvupExpressOrder
     * @param array $data
     * @return void
     */
    private static function updateRvvupExpressSessionData(array $rvvupExpressOrder, array $data)
    {
        $rvvupExpressOrder["cancel_url"] = isset($data["cancel_url"]) ? sanitize_text_field($data["cancel_url"]) : "";
        $rvvupExpressOrder["billing_address"] = [
            "first_name" => isset($data["billing_address"]["first_name"])
                ? sanitize_text_field($data["billing_address"]["first_name"])
                : "",
            "last_name" => isset($data["billing_address"]["last_name"])
                ? sanitize_text_field($data["billing_address"]["last_name"])
                : "",
            "email_address" => isset($data["billing_address"]["email_address"])
                ? sanitize_text_field($data["billing_address"]["email_address"])
                : "",
            "phone_number" => isset($data["billing_address"]["phone_number"])
                ? sanitize_text_field($data["billing_address"]["phone_number"])
                : "",
            "company" => isset($data["billing_address"]["company"])
                ? sanitize_text_field($data["billing_address"]["company"])
                : "",
            "address_line_1" => isset($data["billing_address"]["address_line_1"])
                ? sanitize_text_field($data["billing_address"]["address_line_1"])
                : "",
            "address_line_2" => isset($data["billing_address"]["address_line_2"])
                ? sanitize_text_field($data["billing_address"]["address_line_2"])
                : "",
            "city" => isset($data["billing_address"]["city"])
                ? sanitize_text_field($data["billing_address"]["city"])
                : "",
            "state" => isset($data["billing_address"]["state"])
                ? sanitize_text_field($data["billing_address"]["state"])
                : "",
            "post_code" => isset($data["billing_address"]["post_code"])
                ? sanitize_text_field($data["billing_address"]["post_code"])
                : "",
            "country_code" => isset($data["billing_address"]["country_code"])
                ? sanitize_text_field($data["billing_address"]["country_code"])
                : "",
        ];

        $rvvupExpressOrder["shipping_address"] = [
            "first_name" => isset($data["shipping_address"]["first_name"])
                ? sanitize_text_field($data["shipping_address"]["first_name"])
                : "",
            "last_name" => isset($data["shipping_address"]["last_name"])
                ? sanitize_text_field($data["shipping_address"]["last_name"])
                : "",
            "email_address" => isset($data["shipping_address"]["email_address"])
                ? sanitize_text_field($data["shipping_address"]["email_address"])
                : "",
            "phone_number" => isset($data["shipping_address"]["phone_number"])
                ? sanitize_text_field($data["shipping_address"]["phone_number"])
                : "",
            "company" => isset($data["shipping_address"]["company"])
                ? sanitize_text_field($data["shipping_address"]["company"])
                : "",
            "address_line_1" => isset($data["shipping_address"]["address_line_1"])
                ? sanitize_text_field($data["shipping_address"]["address_line_1"])
                : "",
            "address_line_2" => isset($data["shipping_address"]["address_line_2"])
                ? sanitize_text_field($data["shipping_address"]["address_line_2"])
                : "",
            "city" => isset($data["shipping_address"]["city"])
                ? sanitize_text_field($data["shipping_address"]["city"])
                : "",
            "state" => isset($data["shipping_address"]["state"])
                ? sanitize_text_field($data["shipping_address"]["state"])
                : "",
            "post_code" => isset($data["shipping_address"]["post_code"])
                ? sanitize_text_field($data["shipping_address"]["post_code"])
                : "",
            "country_code" => isset($data["shipping_address"]["country_code"])
                ? sanitize_text_field($data["shipping_address"]["country_code"])
                : "",
        ];

        SessionManager::getInstance()->setRvvupExpressOrder($rvvupExpressOrder);
    }
}
