<?php declare(strict_types=1);

namespace Rvvup\Payments\Controller;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Rvvup\Payments\Contract\PaymentMethodInterface;
use Rvvup\Payments\Service\GatewaySettingsManager;
use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\OrderDetailService;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Traits\MetaDataTrait;

class ReturnEndpoint
{
    use MetaDataTrait;

    public static function execute()
    {
        $rvvupOrderId = $_GET["rvvup-order-id"];
        $orderData = SdkProxy::getOrder($rvvupOrderId);
        $payment = $orderData["payments"][0];
        $paymentStatus = $payment["status"];
        $orderId = $orderData["externalReference"];
        $orderType = $orderData["type"];
        $endpoint = new ReturnEndpoint();
        $logger = LoggerManager::getInstance();
        $metadata = [];
        if (isset($rvvupOrderId)) {
            $metadata["rvvupOrderId"] = $rvvupOrderId;
        }
        if (isset($orderId)) {
            $metadata["orderId"] = $orderId;
            $order = wc_get_order($orderId);
            if (!$order) {
                $logger->error("Return endpoint failed with order ID does not exist ", $metadata);
                wp_die(__("Order ID does not exist", "rvvup-for-woocommerce"), "Rvvup", ["response" => 502]);
            }
        } else {
            if ($orderType === "STANDARD") {
                $logger->error("Return endpoint failed with order was not found for id: ", $metadata);
                wp_die(__("Order ID is required", "rvvup-for-woocommerce"), "Rvvup", ["response" => 400]);
            } elseif ($orderType === "EXPRESS" && $paymentStatus !== PaymentMethodInterface::STATUS_CANCELLED) {
                $logger->error("Express return endpoint failed with order was not found for id: ", $metadata);
                wp_die(__("Invalid order state", "rvvup-for-woocommerce"), "Rvvup", ["response" => 400]);
            }
        }

        if (GatewaySettingsManager::getInstance()->isDebugEnabled()) {
            $logger->debug("redirect payment status is: " . $paymentStatus, $metadata);
        }

        switch ($paymentStatus) {
            case PaymentMethodInterface::STATUS_SUCCEEDED:
                $order->payment_complete($rvvupOrderId);
                OrderDetailService::getInstance()->syncOrderWithRvvupData($order, $orderData);
                if ($payment["captureType"] !== "MANUAL") {
                    self::addCardMetadataNoteToOrder($order, $payment);
                }
                wp_redirect($order->get_checkout_order_received_url());
                break;

            case PaymentMethodInterface::STATUS_PENDING:
            case PaymentMethodInterface::STATUS_REQUIRES_ACTION:
                $order->set_transaction_id($rvvupOrderId);
                $order->add_order_note(
                    "Rvvup returned status `" . $paymentStatus . "`, placing on hold awaiting confirmation"
                );
                $order->set_status("on-hold");
                $order->save();
                wc_add_notice(
                    __(
                        "Your payment is being processed and is pending confirmation. You will receive an email confirmation when the payment is confirmed."
                    ),
                    "notice"
                );
                wp_redirect($order->get_checkout_order_received_url());
                break;

            case PaymentMethodInterface::STATUS_AUTHORIZED:
                $order->set_transaction_id($rvvupOrderId);
                $order->add_order_note("Rvvup returned status `AUTHORIZED`.");
                if (isset($payment["authorizationExpiresAt"]) && $payment["captureType"] === "MANUAL") {
                    // Example: 24th Dec 2023 15:52:28 UTC
                    $formattedExpirationDate = date("jS M Y H:i:s T", strtotime($payment["authorizationExpiresAt"]));
                    $order->add_order_note(
                        "Payment authorization expires at " .
                            $formattedExpirationDate .
                            ". Please navigate to the Rvvup dashboard to manually capture the payment. When the authorization expires, " .
                            "the order will be cancelled and the funds will be returned to the customer."
                    );
                }
                self::addCardMetadataNoteToOrder($order, $payment);
                $order->set_status("on-hold", "Awaiting capture.");
                $order->save();
                wc_add_notice(
                    __(
                        "Your payment is being processed and is pending confirmation. You will receive an email confirmation when the payment is confirmed."
                    ),
                    "notice"
                );
                wp_redirect($order->get_checkout_order_received_url());
                break;

            case PaymentMethodInterface::STATUS_CANCELLED:
                wc_add_notice(__("Payment canceled", "rvvup-for-woocommerce"), "error");
                $data = [
                    "externalReference" => $orderId,
                    "id" => $rvvupOrderId,
                ];

                $endpoint->updateMetaData($orderId, "rvvup_order_payload", json_encode($data));

                wp_redirect(wc_get_checkout_url());
                break;

            case PaymentMethodInterface::STATUS_DECLINED:
                wc_add_notice(
                    __("Payment declined - please try another payment method", "rvvup-for-woocommerce"),
                    "error"
                );
                wp_redirect(wc_get_checkout_url());
                break;

            case PaymentMethodInterface::STATUS_EXPIRED:
                wc_add_notice(
                    __("Payment expired - please try another payment method", "rvvup-for-woocommerce"),
                    "error"
                );
                $data = [
                    "externalReference" => $orderId,
                    "id" => $rvvupOrderId,
                ];

                $endpoint->updateMetaData($orderId, "rvvup_order_payload", json_encode($data));

                wp_redirect(wc_get_checkout_url());
                break;

            case PaymentMethodInterface::STATUS_FAILED:
                wc_add_notice(
                    __("Payment failed - please try another payment method", "rvvup-for-woocommerce"),
                    "error"
                );
                $data = [
                    "externalReference" => $orderId,
                    "id" => $rvvupOrderId,
                ];

                $endpoint->updateMetaData($orderId, "rvvup_order_payload", json_encode($data));

                wp_redirect(wc_get_checkout_url());
                break;

            case PaymentMethodInterface::STATUS_CREATED:
                $data = [
                    "externalReference" => $orderId,
                    "id" => $rvvupOrderId,
                ];

                $endpoint->updateMetaData($orderId, "rvvup_order_payload", json_encode($data));

                wp_redirect(wc_get_checkout_url());
                break;

            default:
                wc_add_notice(
                    __("Sorry, something went wrong while processing the payment.", "rvvup-for-woocommerce"),
                    "error"
                );
                $logger->error("Returned Endpoint fails to unknown payment status", $metadata);
                wp_redirect(wc_get_checkout_url());
        }
    }

    private static function addCardMetadataNoteToOrder(\WC_Order $order, array $payment): void
    {
        try {
            if (
                $order->get_payment_method() == "rvvup_gateway_CARD" ||
                $order->get_payment_method() == "rvvup_gateway_APPLE_PAY"
            ) {
                $data = [];
                $keys = [
                    "cvvResponseCode",
                    "avsAddressResponseCode",
                    "avsPostCodeResponseCode",
                    "eci",
                    "cavv",
                    "acquirerResponseCode",
                    "acquirerResponseMessage",
                ];
                foreach ($keys as $key) {
                    self::populate_card_data($data, $payment, $key);
                }

                if (!empty($data)) {
                    $status = nl2br("Rvvup payment status " . $payment["status"] . "\n Card data: \n");
                    $message = __($status . nl2br(implode("\n", $data)));
                    $order->add_order_note($message, false);
                }
            }
        } catch (\Exception $e) {
            $logger = LoggerManager::getInstance();
            $logger->error("Failed to adding card metadata to order :", [
                "rvvup_payment_data" => $payment,
                "exception" => $e->getMessage(),
            ]);
        }
    }

    /**
     * @param array $data
     * @param array $payment
     * @param string $key
     * @return void
     */
    private static function populate_card_data(array &$data, array $payment, string $key): void
    {
        if (isset($payment[$key])) {
            $value = self::map_card_value($payment[$key]);
            $data[$key] = $key . ": " . $value;
        }
    }

    /**
     * @param string $value
     * @return string
     */
    private static function map_card_value(string $value): string
    {
        switch ($value) {
            case "0":
                if ($value !== "0") {
                    return $value;
                }
                return "0 - Not Given";

            case "1":
                return "1 - Not Checked";

            case "2":
                return "2 - Match";

            case "4":
                return "4 - Not Matched";

            default:
                return $value;
        }
    }
}
