<?php declare(strict_types=1);

namespace Rvvup\Payments\Controller;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Exception;
use Rvvup\Payments\Contract\PaymentMethodInterface;
use Rvvup\Payments\Service\GatewaySettingsManager;
use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\OrderDetailService;
use Rvvup\Payments\Service\OrderStatusManager;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Service\WebhookManager;
use Rvvup\Payments\Traits\EndpointTrait;

class WebhookEndpoint
{
    use EndpointTrait;
    /**
     * Handle incoming webhooks from Rvvup payments backup to update order status
     *
     * @return void
     */
    public static function execute(): void
    {
        $logger = LoggerManager::getInstance();
        $metadata = [];
        // Ensure required params are present
        if (!static::hasRequiredData()) {
            $logger->error("Webhook event failed with missing required params");
            static::returnBadRequestResponse();
            return;
        }

        // If we get a non-supported event type,
        // return a successful response to stop backend from re-sending the webhook.
        if (!static::isSupportedEventType()) {
            $logger->error("Webhook event failed with unsupported event type");
            static::returnSuccessResponse();
            return;
        }

        $merchantId = sanitize_text_field($_GET["merchant_id"]);
        $eventType = sanitize_text_field($_GET["event_type"]);
        $metadata["merchant_id"] = $merchantId;
        $metadata["eventType"] = $eventType;
        $logger = LoggerManager::getInstance();

        try {
            // Ensure configured merchant_id matches request
            if ($merchantId !== GatewaySettingsManager::getInstance()->getMerchantId()) {
                $logger->warning("`merchant_id` from webhook does not match configuration", $metadata);
                static::returnSuccessResponse();
                return;
            }

            switch ($eventType) {
                case "REFUND_COMPLETED":
                    static::handleRefundCompletedEvent();
                    return;
                case "PAYMENT_COMPLETED":
                    static::handlePaymentCompletedEvent();
                    return;
                default:
                    static::returnSuccessResponse();
                    return;
            }
        } catch (Exception $e) {
            $logger->error("Webhook exception: " . $e->getMessage(), $metadata);

            static::returnApplicationErrorResponse();
            return;
        }
    }

    /**
     * Handle business logic for Payment Completed event.
     *
     * @return void
     */
    private static function handlePaymentCompletedEvent(): void
    {
        $logger = LoggerManager::getInstance();

        // Validate request includes order_id param
        if (!static::hasRequiredData(["order_id"])) {
            $logger->info("Webhook was ignored because order_id was missing");
            static::returnBadRequestResponse();
            return;
        }

        $rvvupOrderId = $_GET["order_id"];

        try {
            // Retrieve status data from Rvvup and load related order
            $orderData = SdkProxy::getOrder($rvvupOrderId);

            $paymentStatus = $orderData["payments"][0]["status"];
            $metadata = [];
            $metadata["rvvup_order_id"] = $rvvupOrderId;
            $metadata["payment_status"] = $paymentStatus;
            if (!isset($orderData["externalReference"])) {
                $logger->info("Webhook was ignored for Rvvup order with no associated WooCommerce order", $metadata);
                static::returnSuccessResponse();
                return;
            }
            $order = wc_get_order($orderData["externalReference"]);
            $metadata["wc_order_id"] = $orderData["externalReference"];
            if (!$order) {
                $logger->info("Webhook was ignored because woocommerce order doesn't exist", $metadata);
                static::returnSuccessResponse();
                return;
            }

            // If order is already processed, no action.
            if (
                $order->has_status([
                    "processing",
                    "completed",
                    "refunded",
                    ltrim(OrderStatusManager::STATUS_REFUND_PENDING, "wc-"),
                    ltrim(OrderStatusManager::STATUS_REFUND_FAILED, "wc-"),
                ])
            ) {
                $logger->info("Webhook was ignored because payment already processed", $metadata);
                static::returnSuccessResponse();
                return;
            }
            switch ($paymentStatus) {
                case PaymentMethodInterface::STATUS_SUCCEEDED:
                    $order->payment_complete($orderData["id"]);
                    OrderDetailService::getInstance()->syncOrderWithRvvupData($order, $orderData);
                    break;

                case PaymentMethodInterface::STATUS_EXPIRED:
                case PaymentMethodInterface::STATUS_FAILED:
                case PaymentMethodInterface::STATUS_DECLINED:
                case PaymentMethodInterface::STATUS_CANCELLED:
                    $order->set_status("pending");
                    $order->add_order_note(
                        "Updated to " .
                            $paymentStatus .
                            " by Rvvup webhook, setting to `PENDING` for WooCommerce to cancel"
                    );
                    $order->save();
                    break;

                case PaymentMethodInterface::STATUS_PENDING:
                case PaymentMethodInterface::STATUS_REQUIRES_ACTION:
                    $order->set_status("on-hold");
                    $order->add_order_note(
                        "Updated to " .
                            $paymentStatus .
                            " by Rvvup webhook, setting to `on hold` for WooCommerce to cancel"
                    );
                    $order->save();
                    break;

                case PaymentMethodInterface::STATUS_AUTHORIZATION_EXPIRED:
                    $order->update_status(
                        "cancelled",
                        "Order was cancelled due to the payment authorization expiring. Funds have been released to the customer.",
                        false
                    );
                    break;
            }

            if (!GatewaySettingsManager::getInstance()->isDebugEnabled()) {
                static::returnSuccessResponse();
                return;
            }

            $logger->debug("Webhook data", $metadata);

            static::returnSuccessResponse();
        } catch (Exception $ex) {
            $logger->error("Webhook exception: " . $ex->getMessage());
            $logger->error("Webhook data", [
                "event_type" => "PAYMENT_COMPLETED",
                "id" => $orderData["id"] ?? "FIELD_MISSING",
                "externalReference" => $orderData["externalReference"] ?? "FIELD_MISSING",
                "status" => $paymentStatus ?? "FIELD_MISSING",
            ]);

            static::returnApplicationErrorResponse();
            return;
        }
    }

    /**
     * Handle business logic for refund completed event.
     *
     * @return void
     */
    private static function handleRefundCompletedEvent(): void
    {
        // Validate request includes order_id param
        if (!static::hasRequiredData(["order_id", "refund_id"])) {
            LoggerManager::getInstance()->error("Refund completed event failed with missing required params");
            static::returnBadRequestResponse();
            return;
        }

        $orderId = $_GET["order_id"];
        $refundId = $_GET["refund_id"];

        if (!WebhookManager::getInstance()->refundCompleted($orderId, $refundId)) {
            LoggerManager::getInstance()->error("Refund completed event failed with refund is not completed");
            static::returnApplicationErrorResponse();
        }

        static::returnSuccessResponse();
    }

    /**
     * Validate minimum required params exist & they are not empty.
     * Allow passing params to check, otherwise check default
     *
     * @param array $requiredParams
     * @return bool
     */
    private static function hasRequiredData(array $requiredParams = []): bool
    {
        $defaultRequiredParams = ["merchant_id", "event_type"];

        // Check against default if required params empty.
        if (empty($requiredParams)) {
            $requiredParams = $defaultRequiredParams;
        }

        foreach ($requiredParams as $requiredParam) {
            if (empty($_GET[$requiredParam])) {
                return false;
            }
        }

        return true;
    }

    /**
     * Validate request includes supported event_type
     *
     * @return bool
     */
    private static function isSupportedEventType(): bool
    {
        $supportedEventTypes = ["PAYMENT_COMPLETED", "REFUND_COMPLETED"];

        return in_array($_GET["event_type"] ?? "", $supportedEventTypes, true);
    }
}
