<?php
declare(strict_types=1);

namespace Rvvup\Payments\Gateway;

use Exception;
use Rvvup\Payments\Contract\PaymentMethodInterface;
use Rvvup\Payments\Model\PostTypes\RvvupRefund;
use Rvvup\Payments\Service\GatewaySettingsManager;
use Rvvup\Payments\Service\LoggerManager;
use Rvvup\Payments\Service\RefundManager;
use Rvvup\Payments\Service\RvvupAvailable;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Service\SessionManager;
use Rvvup\Payments\Service\TaxRateCalculator;
use Rvvup\Payments\Traits\MetaDataTrait;
use WC_Order;
use WP_Error;

class Dynamic extends Rvvup
{
    use MetaDataTrait;
    /**
     * The method code.
     *
     * @var string|null
     */
    private $code;

    /**
     * The method summary URL used during checkout.
     *
     * @var string|null
     */
    private $summaryUrl;

    /**
     * The method settings property as provided from Rvvup.
     *
     * @var array|null
     */
    private $apiData;

    /**
     * The method capture type.
     *
     * @var string|null
     */
    private $captureType;

    /**
     * @param string $identifier
     * @param string $title
     * @param string $icon
     * @param string $summaryUrl
     * @param array $apiData
     * @param string captureType
     * @return void
     */
    public function __construct($identifier, $title, $icon, $summaryUrl, $apiData, $captureType)
    {
        $this->supports = ["products", "refunds"];
        $this->code = $identifier;
        $this->id = "rvvup_gateway_$identifier";
        $this->method_title = $title;
        $this->icon = $icon;
        $this->method_description = $title;
        $this->summaryUrl = $summaryUrl;
        parent::__construct();
        $this->title = $title;
        $this->apiData = $apiData;
        $this->captureType = $captureType;
    }

    public function payment_fields()
    {
        if (mb_strtolower($this->code) === "card") {
            if (isset($this->apiData["flow"]) && $this->apiData["flow"] === "INLINE") {
                include RVVUP_PLUGIN_PATH . "Gateway/template/html-checkout-card-inline.php";
                return;
            }
        }
        include RVVUP_PLUGIN_PATH . "Gateway/template/html-checkout-widget.php";
    }

    /**
     * @inheritdoc
     *
     * Check whether rvvup payment methods are available or not for the Cart
     * If any product is Rvvup Restricted and payment method is Clearpay or Zrf
     * If any product is of a product type not allowed.
     *
     * @return bool
     * @throws \Exception
     */
    public function is_available()
    {
        if (!did_action("wp_loaded")) {
            return false;
        }
        $rvvupAvailableService = RvvupAvailable::getInstance();

        // If cards are inline, we need the initialization token to be set, or we can't show the card fields so we hide it.
        if (mb_strtolower($this->code) === "card") {
            if (isset($this->apiData["flow"]) && $this->apiData["flow"] === "INLINE") {
                if (!$this->apiData["initializationToken"]) {
                    return false;
                }
            }
        }
        $total = "0";
        $isOrderPayAjax = isset($_POST["woocommerce_pay"]);
        if (is_checkout_pay_page() || $isOrderPayAjax) {
            global $wp;
            $order_id =
                (is_checkout_pay_page() ? $wp->query_vars["order-pay"] : sanitize_text_field($_POST["order_id"])) ??
                null;
            if ($order_id) {
                $order = wc_get_order($order_id) ?? null;
                if ($order) {
                    $total = (string) round((float) $order->get_total(), 2);
                }
            }
        } elseif (WC()->cart) {
            foreach (WC()->cart->get_cart() as $item) {
                // If payment method is clearpay/zrf and product is Rvvup restricted, payment method is not available.
                if (
                    !$rvvupAvailableService->isRvvupAvailableForProductByProductId(
                        (string) $item["data"]->get_id(),
                        $this->code
                    )
                ) {
                    return false;
                }

                if (!$rvvupAvailableService->isRvvupAvailableForProductByProductType($item["data"]->get_type())) {
                    return false;
                }
            }
            $total = (string) WC()->cart->get_total(null);
        }

        $methods = SdkProxy::getMethods($total);

        foreach ($methods as $method) {
            if ($this->code === $method["name"]) {
                /*
                 * Side Effect: This resets the summary URL as we faced an issue with woocommerce checkout payments,
                 * not updating the info box correctly on the ajax updates.
                 */
                $this->summaryUrl = $method["summaryUrl"];

                // If express payment in the session, pass the relevant query param in the summary URL
                $expressOrder = SessionManager::getInstance()->getRvvupExpressOrder();

                if ($expressOrder !== null && $expressOrder["payment_method_code"] === mb_strtolower($this->code)) {
                    $this->setSummaryUrlMode("express");
                }

                // If a payment method does not have any limits, then the method is available.
                $hasLimits = is_array($method["limits"]);
                if (!$hasLimits) {
                    return true;
                }

                foreach ($method["limits"]["total"] as $threshold) {
                    if (get_woocommerce_currency() === $threshold["currency"]) {
                        return $total <= $threshold["max"] && $total >= $threshold["min"];
                    }
                }

                return false;
            }
        }
        return true;
    }

    /**
     * @param $order_id
     * @return array|mixed
     */
    public function process_payment($order_id)
    {
        $metadata = [];
        $metadata["order_id"] = $order_id;

        if (GatewaySettingsManager::getInstance()->isDebugEnabled()) {
            LoggerManager::getInstance()->debug("Processing payment", $metadata);
        }
        $order = wc_get_order($order_id);
        $order->update_status("wc-pending");

        // If we have a session variable for `rvvup_express_order` this was set from an express payment type.
        if (SessionManager::getInstance()->getRvvupExpressOrder() !== null) {
            return $this->process_express_order($order);
        }

        $variables = $this->create_order_input_from_wc_order($order);
        $response = $this->create_and_process_rvvup_order($variables, $order);
        $metadata["rvvup_order_id"] = $response["id"];
        if (GatewaySettingsManager::getInstance()->isDebugEnabled()) {
            LoggerManager::getInstance()->debug("Processed payment", $metadata);
        }
        return $response;
    }

    public function process_cart_payment($cart)
    {
        $variables = $this->create_order_input_from_wc_cart($cart);
        $response = $this->create_and_process_rvvup_order($variables);

        SessionManager::getInstance()->setRvvupExpressOrder([
            "id" => $response["id"],
            "payment_method_code" => is_string($this->code) ? mb_strtolower($this->code) : null,
        ]);

        return $response;
    }

    /**
     * @param $order_id
     * @param $amount
     * @param $reason
     * @return true|WP_Error
     * @throws \WC_Data_Exception
     */
    public function process_refund($order_id, $amount = null, $reason = "")
    {
        $order = wc_get_order($order_id);

        if ($order === false) {
            return new WP_Error("refund", "Order does not exist");
        }

        $refunds = wc_get_orders([
            "type" => "shop_order_refund",
            "parent" => $order_id,
            "limit" => 1,
        ]);

        // We should get a single result which is the one triggering the Gateway refund request, error if not.
        if (empty($refunds)) {
            return new WP_Error("refund", "No refunds exist for this order");
        }

        /** @var \WC_Order_Refund $refund */
        $refund = $refunds[0];

        try {
            $result = SdkProxy::refundCreate(
                $order->get_transaction_id(),
                $amount,
                $refund->get_currency(),
                "refund-" . $refund->get_id() . "-" . $order->get_order_key(),
                $reason
            );

            // On success, no need to create a Rvvup Refund entity & wait for a webhook.
            if ($result["status"] === Rvvup::REFUND_STATUS_SUCCEEDED) {
                return true;
            }

            // On pending, create Rvvup Refund entity to wait for result via webhook.
            if ($result["status"] === Rvvup::REFUND_STATUS_PENDING) {
                $rvvupRefund = new RvvupRefund();

                $rvvupRefund->set_parent_id($order->get_id());
                $rvvupRefund->set_refund_id($result["id"]);
                $rvvupRefund->set_woocommerce_refund_id($refund->get_id());
                $rvvupRefund->set_amount((string) $amount);
                $rvvupRefund->set_currency($refund->get_currency());
                $rvvupRefund->set_refunded_by($refund->get_refunded_by());
                $rvvupRefund->set_reason($reason ?? "");
                $rvvupRefund->set_status("wc-pending");
                $rvvupRefund->set_is_full_refund(0); // Set full refund placeholder flag, false is default value.

                // Additional data, copied from refund.
                $rvvupRefund->set_prices_include_tax($refund->get_prices_include_tax());
                $rvvupRefund->set_total($refund->get_total());
                $rvvupRefund->set_cart_tax($refund->get_cart_tax());
                $rvvupRefund->set_shipping_total($refund->get_shipping_total());
                $rvvupRefund->set_shipping_tax($refund->get_shipping_tax());
                $rvvupRefund->set_discount_total($refund->get_discount_total());
                $rvvupRefund->set_discount_tax($refund->get_discount_tax());

                $rvvupRefund->save();

                RefundManager::getInstance()->disableRefundNotifications();

                return true;
            }

            // On failed or no status returned (all other cases), return failed.
            return new WP_Error("refund", "Refund Failed");
        } catch (Exception $e) {
            return new WP_Error("refund", $e->getMessage());
        }
    }

    /**
     * Can the order be refunded via this gateway?
     *
     * @param \WC_Order $order Order object.
     * @return bool If false, the automatic refund button is hidden in the UI.
     */
    public function can_refund_order($order)
    {
        // First natively check if we can refund the order, if not, no need for API Call.
        if (!parent::can_refund_order($order) || !$this->isRvvupOrder($order) || !$order->get_transaction_id()) {
            return false;
        }

        try {
            return (bool) SdkProxy::isOrderRefundable($order);
        } catch (Exception $ex) {
            LoggerManager::getInstance()->error(
                "Failed to check whether an order is refundable with message: " . $ex->getMessage(),
                [
                    "order_id" => $order->get_id(),
                    "rvvup_order_id" => $order->get_transaction_id(),
                ]
            );

            return false;
        }
    }

    /**
     * Check if order was payed by rvvup
     * @param $order
     * @return bool
     */
    private function isRvvupOrder($order): bool
    {
        $paymentMethod = $order->get_payment_method();
        return $paymentMethod && stripos($paymentMethod, "rvvup_gateway_") === 0;
    }

    /**
     * Get icon image markup for gateway.
     *
     * @return string icon html
     * @see WC_Payment_Gateway::get_icon()
     */
    public function get_icon()
    {
        $icon = "";

        if ($this->icon) {
            $icon = sprintf(
                '<img src="%s" alt="%s" class="wc-payment-gateway-icon wc-%s-payment-gateway-icon" height="26px" />',
                esc_url($this->icon),
                esc_attr($this->get_title()),
                esc_attr($this->id)
            );
        }

        return apply_filters("woocommerce_gateway_icon", $icon, $this->id);
    }

    /**
     * Get an array of the order's customer data.
     * ToDo: Move these to separate order data builder class.
     *
     * @param \WC_Order $order
     * @return array
     */
    public function getOrderCustomerData(WC_Order $order): array
    {
        return [
            "givenName" => $order->get_billing_first_name(),
            "surname" => $order->get_billing_last_name(),
            "phoneNumber" => $order->get_billing_phone(),
            "email" => $order->get_billing_email(),
        ];
    }

    /**
     * Get the nested api settings property from the string representation of the nested path.
     *
     * @param string $path
     * @return mixed|null
     */
    public function getApiConfig(string $path)
    {
        $path = trim($path);
        $data = $this->apiData;

        // If empty string path, return all data.
        if ($path === "") {
            return $data;
        }

        $keys = explode("/", $path);
        foreach ($keys as $key) {
            if (!isset($data[$key])) {
                return null;
            }

            $data = $data[$key];
        }
        return $data;
    }

    /**
     * @return string
     */
    public function getSummaryUrl(): string
    {
        return $this->summaryUrl ?? "";
    }

    /**
     * @param $variables
     * @return mixed
     * @throws Exception
     */
    private function createOrUpdateOrder($variables)
    {
        $id = $variables["input"]["id"] ?? null;

        if (isset($id)) {
            $orderData = SdkProxy::getOrder($id);
            if (
                isset($orderData["payments"][0]["status"]) &&
                $orderData["payments"][0]["status"] === PaymentMethodInterface::STATUS_CREATED &&
                $variables["input"]["type"] !== "EXPRESS"
            ) {
                $payment = SdkProxy::cancelPayment($orderData["payments"][0]["id"], $orderData["id"]);
                if ($payment["status"] != PaymentMethodInterface::STATUS_CANCELLED) {
                    foreach ($orderData["payments"][0]["summary"]["paymentActions"] as $paymentAction) {
                        if ($paymentAction["type"] == "CANCEL") {
                            $orderData["cancel_url"] = $paymentAction["value"];
                        }
                        return $orderData;
                    }
                }
            }

            if ($orderData["status"] == Rvvup::STATUS_EXPIRED) {
                return $this->createNewOrder($variables["input"]["externalReference"]);
            }

            if ($variables["input"]["type"] == "EXPRESS") {
                unset($variables["input"]["items"]);
            }
            unset($variables["input"]["type"]);
            unset($variables["input"]["idempotencyKey"]);
            return SdkProxy::updateOrder($variables)["data"]["orderUpdate"];
        } else {
            $urlParts = wp_parse_url(get_site_url());
            $variables["input"]["metadata"]["domain"] = $urlParts["host"];

            $variables["input"]["type"] = "V2";
            unset($variables["input"]["method"]);
            $order = SdkProxy::createOrder($variables)["data"]["orderCreate"];
            if ($order["status"] == Rvvup::STATUS_EXPIRED) {
                return $this->createNewOrder($variables["input"]["externalReference"]);
            }
            return $order;
        }
    }

    private function createNewOrder($id)
    {
        $this->deleteMetaData($id, "rvvup_order_payload");

        $order = wc_get_order($id);
        $variables = $this->create_order_input_from_wc_order($order);
        $variables["input"]["idempotencyKey"] = $variables["input"]["idempotencyKey"] . uniqid();
        $variables["input"]["type"] = "V2";

        $urlParts = wp_parse_url(get_site_url());
        $variables["input"]["metadata"]["domain"] = $urlParts["host"];

        $order = SdkProxy::createOrder($variables);

        return $order["data"]["orderCreate"];
    }

    private function create_and_process_rvvup_order($variables, $wcOrder = null)
    {
        $rvvup_order = $this->createOrUpdateOrder($variables);

        if (isset($rvvup_order["cancel_url"]) && $rvvup_order["cancel_url"]) {
            return [
                "result" => "success",
                "id" => $rvvup_order["id"],
                "paymentActions" => [
                    "authorization" => [
                        "redirect_url" => $rvvup_order["cancel_url"],
                    ],
                    "cancel" => [
                        "redirect_url" => $rvvup_order["cancel_url"],
                    ],
                ],
            ];
        }

        if (isset($wcOrder)) {
            $this->updateMetaData($wcOrder->get_id(), "_rvvup_url", $rvvup_order["dashboardUrl"]);
            $this->updateMetaData($wcOrder->get_id(), "_rvvup_title", $this->title);
        }

        $data = [
            "input" => [
                "orderId" => $rvvup_order["id"],
                "method" => $this->code,
                "type" => $variables["input"]["type"],
                "idempotencyKey" => uniqid(),
                "merchantId" => $this->settings["merchant_id"],
                "captureType" => $this->captureType,
            ],
        ];

        $paymentActions = [];
        $redirectUrl = null;
        try {
            $result = SdkProxy::createPayment($data);
            if (isset($result["data"]["paymentCreate"]["summary"]["paymentActions"])) {
                foreach ($result["data"]["paymentCreate"]["summary"]["paymentActions"] as $paymentAction) {
                    if (!isset($paymentActions[mb_strtolower($paymentAction["type"])])) {
                        $paymentActions[mb_strtolower($paymentAction["type"])] = [];
                    }
                    $paymentActions[mb_strtolower($paymentAction["type"])][mb_strtolower($paymentAction["method"])] =
                        $paymentAction["value"];
                }

                if (isset($paymentActions["authorization"]["redirect_url"])) {
                    $redirectUrl = $paymentActions["authorization"]["redirect_url"];
                }
            }

            if (isset($wcOrder)) {
                $this->updateMetaData($wcOrder->get_id(), "rvvup_order_id", $rvvup_order["id"]);
                $this->updateMetaData($wcOrder->get_id(), "rvvup_payment_id", $result["data"]["paymentCreate"]["id"]);
            }

            return [
                "result" => "success",
                "redirect" => $redirectUrl,
                "paymentActions" => $paymentActions,
                "paymentActionsRawJson" => json_encode($paymentActions),
                "id" => $rvvup_order["id"],
            ];
        } catch (Exception $e) {
            if (isset($wcOrder)) {
                $data = [
                    "externalReference" => (string) $wcOrder->get_id(),
                    "id" => $rvvup_order["id"],
                ];
                $this->updateMetaData($wcOrder->get_id(), "rvvup_order_payload", json_encode($data));
            }
            throw $e;
        }
    }

    private function create_order_input_from_wc_order($order, $type = "STANDARD")
    {
        $format_order_item = function ($item) use ($order) {
            /** @var \WC_Product $product */
            $product = $item->get_product();
            $status = $product->get_meta("_rvvup_restricted") === "yes" ? "RESTRICTED" : "ALLOWED";
            $itemPrice = $order->get_item_subtotal($item);
            $itemTotal = $order->get_line_subtotal($item);

            $itemRequest = [
                "sku" => $product->get_sku(),
                "name" => $item["name"],
                "price" => [
                    "amount" => (string) round((float) $itemPrice, 2),
                    "currency" => $order->get_currency(),
                ],
                "priceWithTax" => [
                    "amount" => (string) round((float) wc_get_price_including_tax($product), 2),
                    "currency" => $order->get_currency(),
                ],
                "tax" => [
                    "amount" => (string) round(wc_get_price_including_tax($product) - $itemPrice, 2),
                    "currency" => $order->get_currency(),
                ],
                "quantity" => (string) $item["quantity"],
                "total" => [
                    "amount" => (string) round((float) $itemTotal, 2),
                    "currency" => $order->get_currency(),
                ],
                "restriction" => $status, // ALLOWED, RESTRICTED or PROHIBITED
            ];

            $taxRate = TaxRateCalculator::getInstance()->getItemTaxRate($order, $product);
            if ($taxRate !== null) {
                $itemRequest["taxRate"] = $taxRate;
            }
            return $itemRequest;
        };
        $formatted_order_items = array_values(array_map($format_order_item, $order->get_items()));

        $discountTotal = $order->get_discount_total();
        if (empty($discountTotal)) {
            $discountTotal = "0.00";
        }

        $shippingTotal = $order->get_shipping_total();
        if (empty($shippingTotal)) {
            $shippingTotal = "0.00";
        }

        $taxTotal = $order->get_total_tax();
        if (empty($taxTotal)) {
            $taxTotal = "0.00";
        }

        //new idempotencyKey orderLevel
        // two pages -> two modals -> cancel both -> start one + close tab -> return cancel url!
        $variables = [
            "externalReference" => (string) $order->get_id(),
            "type" => $type,
            "idempotencyKey" => (string) $order->get_id(),
            "total" => [
                "amount" => (string) round((float) $order->get_total(), 2),
                "currency" => $order->get_currency(),
            ],
            "discountTotal" => [
                "amount" => (string) round((float) $discountTotal, 2),
                "currency" => $order->get_currency(),
            ],
            "shippingTotal" => [
                "amount" => (string) round((float) $shippingTotal, 2),
                "currency" => $order->get_currency(),
            ],
            "taxTotal" => [
                "amount" => (string) round((float) $taxTotal, 2),
                "currency" => $order->get_currency(),
            ],
            "customer" => $this->getOrderCustomerData($order),
            "billingAddress" => $this->getOrderBillingAddressData($order),
            "requiresShipping" => $order->needs_shipping_address(),
            "items" => $formatted_order_items,
        ];

        if ($order->needs_shipping_address()) {
            $variables["shippingAddress"] = $this->getOrderShippingAddressData($order);
        }

        $payload = $this->getMetaData($order->get_id(), "rvvup_order_payload");
        if (!empty($payload)) {
            $payload = json_decode($payload, true);
            $variables["id"] = $payload["id"];
            $variables["externalReference"] = (string) $payload["externalReference"];
            $variables["merchantId"] = $this->settings["merchant_id"];
        } else {
            $variables["merchant"] = [
                "id" => $this->settings["merchant_id"],
            ];
            $variables["redirectToStoreUrl"] = $this->settings["callback_url"];
        }

        return ["input" => $variables];
    }

    private function create_order_input_from_wc_cart($cart)
    {
        $currency = get_woocommerce_currency();
        $format_cart_item = function ($item) use ($cart, $currency) {
            /** @var \WC_Product $product */
            $product = $item["data"];
            $status = $product->get_meta("_rvvup_restricted") === "yes" ? "RESTRICTED" : "ALLOWED";
            $itemPrice = $product->get_price();
            $itemTotal = (float) $itemPrice * $item["quantity"];
            return [
                "sku" => $product->get_sku(),
                "name" => $product->get_name(),
                "price" => [
                    "amount" => (string) round((float) $itemPrice, 2),
                    "currency" => $currency,
                ],
                "priceWithTax" => [
                    "amount" => (string) round((float) wc_get_price_including_tax($product), 2),
                    "currency" => $currency,
                ],
                "tax" => [
                    "amount" => (string) round(wc_get_price_including_tax($product) - $itemPrice, 2),
                    "currency" => $currency,
                ],
                "quantity" => (string) $item["quantity"],
                "total" => [
                    "amount" => (string) round($itemTotal, 2),
                    "currency" => $currency,
                ],
                "restriction" => $status, // ALLOWED, RESTRICTED or PROHIBITED
            ];
        };

        $formatted_cart_items = array_values(array_map($format_cart_item, $cart->get_cart_contents()));

        $discountTotal = $cart->get_total_discount();
        if (empty($discountTotal)) {
            $discountTotal = "0.00";
        }

        $taxTotal = (string) $cart->get_total_tax();
        if (empty($taxTotal)) {
            $taxTotal = "0.00";
        }
        $customer = WC()->customer;
        $customerInput = null;
        $billingInput = null;
        $shippingInput = null;
        if (is_user_logged_in() && $customer instanceof \WC_Customer) {
            if ($this->has_full_customer($customer)) {
                $customerInput = [
                    "givenName" => $customer->get_first_name(),
                    "surname" => $customer->get_last_name(),
                    "phoneNumber" => $customer->get_billing_phone(),
                    "email" => $customer->get_email(),
                ];
            }
            if ($this->has_full_billing_address($customer)) {
                $billingInput = [
                    "name" => $customer->get_billing_first_name() . " " . $customer->get_billing_last_name(),
                    "phoneNumber" => $customer->get_billing_phone(),
                    "company" => $customer->get_billing_company(),
                    "line1" => $customer->get_billing_address_1(),
                    "line2" => $customer->get_billing_address_2(),
                    "city" => $customer->get_billing_city(),
                    "state" => $customer->get_billing_state(),
                    "postcode" => $customer->get_billing_postcode(),
                    "countryCode" => $customer->get_billing_country(),
                ];
            }
            if ($cart->needs_shipping() && $this->has_full_shipping_address($customer)) {
                $shippingInput = [
                    "name" => $customer->get_shipping_first_name() . " " . $customer->get_shipping_last_name(),
                    "company" => $customer->get_shipping_company(),
                    "line1" => $customer->get_shipping_address_1(),
                    "line2" => $customer->get_shipping_address_2(),
                    "city" => $customer->get_shipping_city(),
                    "state" => $customer->get_shipping_state(),
                    "postcode" => $customer->get_shipping_postcode(),
                    "countryCode" => $customer->get_shipping_country(),
                ];
                // get_shipping_phone() is only supported on Woocommerce 5.6.0 or higher.
                if (method_exists($customer, "get_shipping_phone")) {
                    $shippingInput["phoneNumber"] = $customer->get_shipping_phone();
                }
            }
        }

        $data = [
            "type" => "EXPRESS",
            "total" => [
                "amount" => (string) round((float) $cart->get_total(null), 2),
                "currency" => $currency,
            ],
            "discountTotal" => [
                "amount" => (string) round((float) $discountTotal, 2),
                "currency" => $currency,
            ],
            "shippingTotal" => [
                "amount" => (string) round((float) $cart->get_shipping_total(), 2),
                "currency" => $currency,
            ],
            "taxTotal" => [
                "amount" => (string) round((float) $taxTotal, 2),
                "currency" => $currency,
            ],
            "customer" => $customerInput,
            "billingAddress" => $billingInput,
            "shippingAddress" => $shippingInput,
            "requiresShipping" => $cart->needs_shipping(),
            "items" => $formatted_cart_items,
            "redirectToStoreUrl" => $this->settings["callback_url"],
            "merchant" => [
                "id" => $this->settings["merchant_id"],
            ],
        ];

        return ["input" => $data];
    }

    private function has_full_shipping_address(\WC_Customer $customer)
    {
        return $this->field_exists($customer->get_shipping_first_name()) &&
            $this->field_exists($customer->get_shipping_last_name()) &&
            $this->field_exists($customer->get_shipping_address_1()) &&
            $this->field_exists($customer->get_shipping_city()) &&
            $this->field_exists($customer->get_shipping_postcode()) &&
            $this->field_exists($customer->get_shipping_country());
    }

    private function has_full_billing_address(\WC_Customer $customer)
    {
        return $this->field_exists($customer->get_billing_first_name()) &&
            $this->field_exists($customer->get_billing_last_name()) &&
            $this->field_exists($customer->get_billing_address_1()) &&
            $this->field_exists($customer->get_billing_city()) &&
            $this->field_exists($customer->get_billing_postcode()) &&
            $this->field_exists($customer->get_billing_country());
    }

    private function has_full_customer(\WC_Customer $customer)
    {
        return $this->field_exists($customer->get_first_name()) &&
            $this->field_exists($customer->get_last_name()) &&
            $this->field_exists($customer->get_email()) &&
            $this->field_exists($customer->get_billing_phone());
    }

    private function field_exists($field)
    {
        return isset($field) && strlen(trim($field)) > 0;
    }

    /**
     * Get an array of the order's billing address data.
     * ToDo: Move these to separate order data builder class.
     *
     * @param \WC_Order $order
     * @return array
     */
    private function getOrderBillingAddressData(WC_Order $order)
    {
        return [
            "name" => $order->get_formatted_billing_full_name(),
            "phoneNumber" => $order->get_billing_phone(),
            "company" => $order->get_billing_company(),
            "line1" => $order->get_billing_address_1(),
            "line2" => $order->get_billing_address_2(),
            "city" => $order->get_billing_city(),
            "state" => $order->get_billing_state(),
            "postcode" => $order->get_billing_postcode(),
            "countryCode" => $order->get_billing_country(),
        ];
    }

    /**
     * Get an array of the order's shipping address data.
     * ToDo: Move these to separate order data builder class.
     *
     * @param \WC_Order $order
     * @return array
     */
    private function getOrderShippingAddressData(WC_Order $order)
    {
        $data = [
            "name" => $order->get_formatted_shipping_full_name(),
            "company" => $order->get_shipping_company(),
            "line1" => $order->get_shipping_address_1(),
            "line2" => $order->get_shipping_address_2(),
            "city" => $order->get_shipping_city(),
            "state" => $order->get_shipping_state(),
            "postcode" => $order->get_shipping_postcode(),
            "countryCode" => $order->get_shipping_country(),
        ];

        // get_shipping_phone() is only supported on Woocommerce 5.6.0 or higher.
        if (method_exists($order, "get_shipping_phone")) {
            $data["phoneNumber"] = $order->get_shipping_phone();
        }

        return $data;
    }

    /**
     * Set the summary URL mode from the corresponding param.
     *
     * @param string $mode
     * @return void
     */
    private function setSummaryUrlMode(string $mode)
    {
        if (!in_array($mode, ["express", "standard"], true)) {
            return;
        }

        $parsedSummaryUrl = wp_parse_url($this->summaryUrl);

        // If for any reason we don't get a full array of URL components, return without logging errors.
        if (!is_array($parsedSummaryUrl)) {
            return;
        }

        // Set the base summary URL without the query parameters. We always expect scheme, host, path.
        $baseSummaryUrl = $parsedSummaryUrl["scheme"] . "://" . $parsedSummaryUrl["host"] . $parsedSummaryUrl["path"];

        // If query not set, add it and return parsed URL.
        if (!isset($parsedSummaryUrl["query"])) {
            $parsedSummaryUrl["query"] = "mode=" . $mode;

            // We expect a summary URL without port, username, pass.
            $this->summaryUrl =
                $baseSummaryUrl .
                "?" .
                $parsedSummaryUrl["query"] .
                (isset($parsedSummaryUrl["fragment"]) ? "#" . $parsedSummaryUrl["fragment"] : "");

            return;
        }

        // Instead, if set, parse the query string to an array to get all the parameters.
        wp_parse_str($parsedSummaryUrl["query"], $queryParams);

        // If mode is already set in the query params, do nothing.
        if (isset($queryParams["mode"])) {
            return;
        }

        // Otherwise, add it.
        $queryParams["mode"] = $mode;

        // Build the query and specify RFC for spaces as percent encoding `%20`.
        $this->summaryUrl =
            $baseSummaryUrl .
            "?" .
            http_build_query($queryParams) .
            (isset($parsedSummaryUrl["fragment"]) ? "#" . $parsedSummaryUrl["fragment"] : "");
    }

    private function process_express_order($order)
    {
        $id = SessionManager::getInstance()->getRvvupExpressOrder()["id"];
        SessionManager::getInstance()->unsetRvvupExpressOrder();
        $data = [
            "externalReference" => (string) $order->get_id(),
            "id" => $id,
        ];

        $this->updateMetaData($order->get_id(), "rvvup_order_payload", json_encode($data));

        $variables = $this->create_order_input_from_wc_order($order, "EXPRESS");
        $data = $this->createOrUpdateOrder($variables);

        return [
            "result" => "success",
            "id" => $data["id"],
            "paymentActions" => [
                "authorization" => [
                    "redirect_url" => $data["redirectToCheckoutUrl"],
                ],
                "cancel" => [
                    "redirect_url" => wc_get_checkout_url(),
                ],
            ],
        ];
    }
}
