<?php declare(strict_types=1);

namespace Rvvup\Payments\Gateway;

use Exception;
use Rvvup\Payments\Model\SimpleJWT;
use Rvvup\Payments\Service\EventManager;
use Rvvup\Payments\Service\SdkProxy;

class Rvvup extends \WC_Payment_Gateway
{
    const ID = "rvvup_gateway";
    /** @var string The needs to do something to finalise the transaction */
    const STATUS_REQUIRES_ACTION = "REQUIRES_ACTION";
    /** @var string Payment has started but has not been captured */
    const STATUS_CREATED = "CREATED";
    /** @var string Expired, order should be cancelled */
    const STATUS_EXPIRED = "EXPIRED";
    /** @var string Failed, order should be cancelled */
    const STATUS_FAILED = "FAILED";
    /** @var string Customer has clicked place order but has not finished all steps  */
    const STATUS_PENDING = "PENDING";
    /** @var string Payment was completed sucessfully */
    const STATUS_SUCCEEDED = "SUCCEEDED";
    /** @var string Customer aborted the payment */
    const STATUS_CANCELLED = "CANCELLED";
    /** @var string Customer's issuing bank rejected the payment */
    const STATUS_DECLINED = "DECLINED";

    /**
     * Refund statuses returned via the API
     */
    public const REFUND_STATUS_FAILED = "FAILED";
    public const REFUND_STATUS_PENDING = "PENDING";
    public const REFUND_STATUS_SUCCEEDED = "SUCCEEDED";

    public function __construct()
    {
        $this->id = !empty($this->id) ? $this->id : self::ID;
        $this->has_fields = true;
        $this->title = __("Rvvup", "rvvup-for-woocommerce");
        $this->method_description = __("Payments via Rvvup", "rvvup-for-woocommerce");
        $this->init_form_fields();
        $this->init_settings();

        if (!has_action("woocommerce_update_options_payment_gateways_" . $this->id)) {
            add_action("woocommerce_update_options_payment_gateways_" . $this->id, [$this, "process_admin_options"]);
        }

        add_action("woocommerce_update_option", [$this, "rvvupToggleDisabled"], 10, 1);
        add_filter("wc_rvvup_settings_nav_tabs", [$this, "admin_nav_tab"]);
        add_action("woocommerce_settings_checkout", [$this, "output_settings_nav"], 1);
        add_action(
            "woocommerce_generate_rvvup_account_dashboard_link_html",
            [$this, "getAccountDashboardLinkHtml"],
            10,
            3
        );
        add_action("woocommerce_generate_rvvup_log_link_html", [$this, "getLogLinkHtml"], 10, 3);
    }

    /**
     * @return string
     */
    public function getAccountDashboardLinkHtml(...$args)
    {
        // Array index 2 should have the data we need from the arguments.
        $accountDashboardArguments = $args["2"] ?? $this->getAccountDashboardLinkArguments();

        return '<tr valign="top">
            <th scope="row" class="titledesc"></th>
            <td class="forminp">
                <a href="' .
            $accountDashboardArguments["default"] .
            '"
                class="button button-secondary"
                target="_blank">' .
            $accountDashboardArguments["title"] .
            '</a>
            </td>
        </tr>';
    }

    public function getLogLinkHtml(...$args)
    {
        $url = get_site_url(null, "wp-admin/admin-ajax.php?action=rvvup_log");
        $html = <<<HTML
        <tr valign="top">
            <th scope="row" class="titledesc">
                Debug Log
            </th>
            <td class="forminp">
                <a href="$url">Download</a>
            </td>
        </tr>
HTML;
        return $html;
    }
    public function output_settings_nav()
    {
        if (isset($_GET["page"]) && isset($_GET["section"])) {
            $is_rvvup_section = $_GET["page"] === "wc-settings" && $_GET["section"] === "rvvup_gateway";
            if ($is_rvvup_section) {
                include_once RVVUP_PLUGIN_PATH . "/Gateway/template/html-settings-nav.php";
            }
        }
    }

    public function admin_nav_tab($tabs)
    {
        $tabs["main"] = "Rvvup";
        return $tabs;
    }

    public function init_form_fields()
    {
        $this->form_fields = [
            "account_dashboard_link" => $this->getAccountDashboardLinkArguments(),
            "enabled" => [
                "title" => __("Enable/Disable", "rvvup-for-woocommerce"),
                "type" => "checkbox",
                "label" => __("Enable Rvvup Payments", "rvvup-for-woocommerce"),
                "default" => "no",
            ],
            "jwt" => [
                "title" => __("API Keys", "rvvup-for-woocommerce"),
                "type" => "password",
            ],
            "environment" => [
                "title" => __("Current Environment", "rvvup-for-woocommerce"),
                "type" => "text",
                "disabled" => true,
            ],
            "show_if_unavailable" => [
                "title" => "Show if unavailable",
                "description" => "If Rvvup can not be contacted show a disabled payment method?",
                "type" => "select",
                "default" => "1",
                "options" => [
                    0 => "No",
                    1 => "Yes",
                ],
            ],
            "restriction_message_pdp" => [
                "title" => "PDP Restriction Message",
                "description" =>
                    "If the product has Rvvup payment restrictions what message should be shown on the product page?",
                "type" => "text",
                "default" => "This item has restrictions so not all payment methods may be available",
            ],
            "restriction_message_checkout_enabled" => [
                "title" => __("Show restriction message in checkout", "rvvup-for-woocommerce"),
                "type" => "checkbox",
                "label" => __("Show restriction message in checkout", "rvvup-for-woocommerce"),
                "default" => "yes",
            ],
            "restriction_message_checkout" => [
                "title" => "Checkout Restriction Message",
                "description" =>
                    "If the product has Rvvup payment restrictions what message should be shown on the checkout?",
                "type" => "text",
                "default" => "You have some restricted items in your cart, not all payment methods may be available",
            ],
            "mobile_position" => [
                "title" => "Mobile modal position",
                "type" => "select",
                "default" => "bottom",
                "options" => [
                    "bottom" => "Bottom",
                    "left" => "Left",
                    "right" => "Right",
                ],
            ],
            "unavailable_error_message" => [
                "title" => "Error Message",
                "description" => "If Rvvup can not be contacted what error message should be shown?",
                "type" => "text",
                "default" => "Rvvup payment is currently not available",
            ],
            "debug" => [
                "title" => __("Enable Debug Mode", "rvvup-for-woocommerce"),
                "type" => "select",
                "default" => 0,
                "options" => [
                    0 => "No",
                    1 => "Yes",
                ],
            ],
            "enabled_product_types" => [
                "title" => __("Enable Product Types", "rvvup-for-woocommerce"),
                "description" =>
                    __("Select which product types Rvvup should be enabled for.", "rvvup-for-woocommerce") .
                    "<br />" .
                    __("Select Simple products as well, if Grouped products are selected.", "rvvup-for-woocommerce") .
                    "<br />" .
                    __(
                        "Select External/Affiliate products only if the external resource is set up with Rvvup payments.",
                        "rvvup-for-woocommerce"
                    ),
                "type" => "multiselect",
                "class" => "wc-enhanced-select",
                "default" => $this->getDefaultProductTypes(),
                "options" => function_exists("wc_get_product_types")
                    ? wc_get_product_types()
                    : $this->getDefaultProductTypes(),
            ],
            "log_link" => [
                "type" => "rvvup_log_link",
                "title" => "Download Log File",
            ],
        ];
    }

    public function init_settings()
    {
        $origId = $this->id;
        $this->id = self::ID;
        parent::init_settings();
        if (empty($this->settings["jwt"])) {
            return;
        }
        if (!isset($this->settings["enabled_product_types"]) || !is_array($this->settings["enabled_product_types"])) {
            $this->settings["enabled_product_types"] = $this->form_fields["enabled_product_types"]["default"];
        }
        $jwt = SimpleJWT::getPayload($this->settings["jwt"]);
        if (!$jwt) {
            return;
        }
        $this->id = $origId;

        $this->settings[SdkProxy::BACKEND_API_URL] = $jwt->aud ?? null;
        $this->settings[SdkProxy::MERCHANT_ID] = $jwt->merchantId ?? null;
        $this->settings[SdkProxy::AUTH_TOKEN] = base64_encode($jwt->username . ":" . $jwt->password);
        $this->settings["callback_url"] = get_site_url(null, "?wc-api=rvvup-return");
        if (isset($jwt->live)) {
            switch ($jwt->live) {
                case true:
                    $environment = "LIVE";
                    break;
                case false:
                    $environment = "TEST";
                    break;
                default:
                    $environment = "UNKNOWN";
            }
            $this->settings["environment"] = $environment;
        }
    }

    public function is_available()
    {
        // Prevent fallback method showing in the checkout if the gateway is disabled
        if (!wc_string_to_bool($this->settings["enabled"])) {
            return false;
        }
        return true;
    }

    public function needs_setup()
    {
        $merchantId = $this->get_option(SdkProxy::MERCHANT_ID);
        $authToken = $this->get_option(SdkProxy::AUTH_TOKEN);
        return !($merchantId && $authToken);
    }

    public function process_admin_options()
    {
        $data = $this->get_post_data();
        if (
            isset($data["woocommerce_rvvup_gateway_jwt"]) &&
            !SimpleJWT::isValid($data["woocommerce_rvvup_gateway_jwt"])
        ) {
            \WC_Admin_Settings::add_error("Invalid JWT entered");
            //unset($_POST["woocommerce_rvvup_gateway_jwt"]);
            return false;
        }

        // Load existing settings.
        $this->init_settings();

        // Validate if payment gateway setting was disabled (if disabled, value is not set) and inform Rvvup via the API.
        if (
            isset($this->settings["enabled"]) &&
            !isset($data["woocommerce_rvvup_gateway_enabled"]) &&
            wc_string_to_bool($this->settings["enabled"])
        ) {
            EventManager::getInstance()->dispatchPluginDisabledEvent("Woocommerce Payment Method deactivated");
        }

        parent::process_admin_options();
        $this->init_settings();
        // Save again so that the JWT data is loaded from the DB
        $saved = parent::process_admin_options();
        try {
            $url = get_site_url(null, "?wc-api=rvvup-webhook");
            $connection = SdkProxy::registerWebhook($url);
        } catch (Exception $e) {
            $connection = false;
            $error = "Unable to connect to Rvvup, " . $e->getMessage();
        }
        if ($connection) {
            \WC_Admin_Settings::add_message("Connection to Rvvup successful, webhook updated");
        } else {
            $error = "Unable to connect to Rvvup, Please check credentials";
            \WC_Admin_Settings::add_error($error);
        }
        return $saved;
    }

    /**
     * @return array
     */
    private function getAccountDashboardLinkArguments()
    {
        return [
            "type" => "rvvup_account_dashboard_link",
            "title" => __("Rvvup Dashboard", "rvvup-for-woocommerce"),
            "default" => "https://dashboard.rvvup.com",
        ];
    }

    /**
     * Get the woocommerce default product types, copied from wc_get_product_types()
     * Do not include Affiliate/External products by default, as they link to an external URL which might not have Rvvup payments set.
     * We don't need the local label, but only the keys as the array values,
     * because that's what's normally saved in the db.
     *
     * @return array
     */
    private function getDefaultProductTypes()
    {
        return ["simple", "grouped", "variable"];
    }

    /**
     * Perform actions when rvvup is disabled via the WooCommerce Payments page toggle functionality.
     * Hooks into the AJAX request.
     *
     * @param array $option
     * @return void
     */
    public function rvvupToggleDisabled($option): void
    {
        // If no admin area, no action.
        if (!is_admin()) {
            return;
        }

        // If not Rvvup updated or not Gateway toggled enabled or Rvvup wno action.
        if (
            !isset($option["id"], $_POST["gateway_id"], $_POST["action"]) ||
            $option["id"] !== $this->get_option_key() ||
            wc_clean(wp_unslash($_POST["gateway_id"])) !== $this->id ||
            wc_clean(wp_unslash($_POST["action"])) !== "woocommerce_toggle_gateway_enabled"
        ) {
            return;
        }

        // The action hook fires before the payment is disabled in the AJAX request.
        // So if current setting is disabled, it will be changing to enabled, so no need to inform Rvvup via the API.
        // If currently enabled, it will be disabled, so inform via the API.
        if (!wc_string_to_bool($this->get_option("enabled"))) {
            return;
        }

        EventManager::getInstance()->dispatchPluginDisabledEvent("Woocommerce Payment Method deactivated");
    }
}
