<?php declare(strict_types=1);

namespace Rvvup\Payments\Model;

use Exception;
use Rvvup\Payments\Controller\CardSubmitCallbackEndpoint;
use Rvvup\Payments\Controller\ExpressDeleteEndpoint;
use Rvvup\Payments\Controller\ExpressEndpoint;
use Rvvup\Payments\Controller\ExpressUpdateEndpoint;
use Rvvup\Payments\Controller\PayOrderEndpoint;
use Rvvup\Payments\Lib\Versions;
use Rvvup\Payments\Service\SdkProxy;
use Rvvup\Payments\Service\SessionManager;
use WC_AJAX;
use WC_Product;
use Rvvup\Payments\Service\GatewaySettingsManager;

class AssetLoader
{
    /**
     * Load Rvvup Payment methods assets where required (checkout & product pages)
     *
     * @return void
     */
    public static function load()
    {
        wp_enqueue_script(
            "rvvup_payment_parameters_js",
            plugins_url("assets/js/parameters.js", RVVUP_PLUGIN_FILE),
            [],
            Versions::getPluginVersion(),
            true
        );
        if (!AssetLoader::canLoadFrontendAssets()) {
            wp_localize_script("rvvup_payment_parameters_js", "rvvup_parameters", ["methods" => []]);
            return;
        }
        $methods = [];
        try {
            $methods = SdkProxy::getMethods((string) self::getTotal());
        } catch (Exception $ignored) {
        }
        // First load the payment methods assets in the header, as provided by Rvvup.
        self::loadPaymentMethodAssets($methods);
        // Then load the plugin assets.
        self::loadRvvupAssets($methods);
    }

    public static function loadAssetsForBlocks(): array
    {
        wp_enqueue_script(
            "rvvup_payment_parameters_js",
            plugins_url("assets/js/parameters.js", RVVUP_PLUGIN_FILE),
            [],
            Versions::getPluginVersion(),
            true
        );

        $asset_file = include RVVUP_PLUGIN_PATH . "assets/js/blocks/index.asset.php";
        wp_register_script(
            "rvvup-blocks-script",
            plugins_url("assets/js/blocks/index.js", RVVUP_PLUGIN_FILE),
            array_merge(["rvvup_payment_parameters_js"], $asset_file["dependencies"]),
            $asset_file["version"],
            true
        );
        return ["rvvup-blocks-script"];
    }

    /**
     * Load admin area assets.
     *
     * @return void
     */
    public static function admin(): void
    {
        // Load styles.
        wp_enqueue_style(
            "rvvup_payment_admin_css",
            plugins_url("assets/css/admin.css", RVVUP_PLUGIN_FILE),
            [],
            Versions::getPluginVersion()
        );

        wp_enqueue_script(
            "rvvup_admin_js",
            plugins_url("assets/js/admin.js", RVVUP_PLUGIN_FILE),
            ["jquery"],
            Versions::getPluginVersion()
        );
        wp_enqueue_script(
            "rvvup_payment_parameters_js",
            plugins_url("assets/js/parameters.js", RVVUP_PLUGIN_FILE),
            [],
            Versions::getPluginVersion(),
            true
        );
        wp_localize_script("rvvup_payment_parameters_js", "rvvup_parameters", ["methods" => []]);
    }

    public static function addRvvupAttributes($tag, $assetName)
    {
        $rvvupAttributes = wp_scripts()->get_data($assetName, "rvvup-attributes");
        if ($rvvupAttributes) {
            $attributes = "";
            foreach ($rvvupAttributes as $key => $value) {
                // If no value provided, just set the key
                if ($value === null) {
                    $attributes .= " $key ";
                    continue;
                }

                $attributes .= " $key=\"$value\" ";
            }
            $tag = str_replace(" src", " $attributes src", $tag);
        }
        return $tag;
    }

    private static function canLoadFrontendAssets(): bool
    {
        return is_checkout() || is_cart() || is_product();
    }

    private static function loadRvvupAssets($methods): void
    {
        // Load styles.
        wp_enqueue_style(
            "rvvup_payment_css",
            plugins_url("assets/css/styles.css", RVVUP_PLUGIN_FILE),
            [],
            Versions::getPluginVersion()
        );

        global $wp;

        $rvvupParameters = [
            "orderId" => $wp->query_vars["order-pay"] ?? 0,
            "endpoints" => self::getEndpointsParameters(),
            "urls" => self::getSiteUrls(),
            "express" => self::getExpressPaymentParameters(),
            "i18n" => self::getLocalizationParameters(),
            "methods" => self::getRvvupPaymentMethodsParameters($methods),
            "settings" => self::getRvvupPaymentMethodsSettings($methods),
        ];

        wp_localize_script("rvvup_payment_parameters_js", "rvvup_parameters", $rvvupParameters);

        wp_enqueue_script("rvvup-core-sdk", GatewaySettingsManager::getInstance()->getJsSdkUrl(), [], null);
        wp_register_script("rvvup-sdk", false);
        wp_enqueue_script("rvvup-sdk");
        wp_add_inline_script(
            "rvvup-sdk",
            'if(window.Rvvup){
                      window.rvvup_sdk = window.Rvvup({
                        publishableKey: \'' .
                GatewaySettingsManager::getInstance()->getPublishableKey() .
                '\'
                      });
                    }'
        );
        // Load paypal helper script
        wp_enqueue_script(
            "rvvup_payment_paypal_js",
            plugins_url("assets/js/paypal.js", RVVUP_PLUGIN_FILE),
            ["rvvup_payment_parameters_js"],
            Versions::getPluginVersion(),
            true
        );

        // Load rest of the scripts if blocks not enabled
        if (method_exists("\Automattic\WooCommerce\Blocks\Utils\CartCheckoutUtils", "is_checkout_block_default")) {
            if (\Automattic\WooCommerce\Blocks\Utils\CartCheckoutUtils::is_checkout_block_default()) {
                return;
            }
        }

        wp_enqueue_script(
            "rvvup_payment_js",
            plugins_url("assets/js/scripts.js", RVVUP_PLUGIN_FILE),
            ["jquery", "rvvup_payment_parameters_js", "rvvup_payment_paypal_js"],
            Versions::getPluginVersion(),
            true
        );
    }

    private static function loadPaymentMethodAssets($methods): void
    {
        foreach ($methods as $method) {
            self::enqueueRequiredAssets(
                "rvvup-" . $method["name"],
                isset($method["settings"]["assets"]) && is_array($method["settings"]["assets"])
                    ? $method["settings"]["assets"]
                    : []
            );
        }
        add_filter("script_loader_tag", [self::class, "addRvvupAttributes"], 10, 2);
    }

    /**
     * Get total, either of product page or cart total.
     *
     * @return float|string
     */
    private static function getTotal()
    {
        if (is_product()) {
            $product = wc_get_product();
            if ($product instanceof WC_Product) {
                return $product->get_price();
            }
        }

        $cart = WC()->cart;

        return $cart !== null ? $cart->get_total(null) : "0";
    }

    /**
     * @param string $assetName
     * @param array $assets
     * @return void
     */
    private static function enqueueRequiredAssets(string $assetName, array $assets): void
    {
        $assetsCount = count($assets);
        foreach ($assets as $index => $assetToLoad) {
            if (!isset($assetToLoad["assetType"], $assetToLoad["url"])) {
                continue;
            }

            if ($assetsCount > 1) {
                $assetName .= "-" . $index;
            }

            if ($assetToLoad["assetType"] === "SCRIPT") {
                wp_enqueue_script($assetName, $assetToLoad["url"], [], null);
                wp_script_add_data($assetName, "rvvup-attributes", $assetToLoad["attributes"] ?? []);
            }
        }
    }

    /**
     * Get the localization (i18n) parameters for the rvvup_parameters js object.
     *
     * @return array
     */
    private static function getLocalizationParameters(): array
    {
        return [
            "generic" => __("Something went wrong, try again!", "rvvup-for-woocommerce"),
            "no_variation_selected" => __("Please select a variation to continue", "rvvup-for-woocommerce"),
            "invalid_quantity" => __("No items have been selected", "rvvup-for-woocommerce"),
        ];
    }

    /**
     * Get the endpoint parameters for the rvvup_parameters js object.
     *
     * @return array
     */
    private static function getEndpointsParameters(): array
    {
        return [
            "payOrder" => WC_AJAX::get_endpoint(PayOrderEndpoint::ENDPOINT),
            "express" =>
                WC_AJAX::get_endpoint(ExpressEndpoint::ENDPOINT) .
                "&nonce=" .
                wp_create_nonce(ExpressEndpoint::ENDPOINT),
            "expressUpdate" =>
                WC_AJAX::get_endpoint(ExpressUpdateEndpoint::ENDPOINT) .
                "&nonce=" .
                wp_create_nonce(ExpressUpdateEndpoint::ENDPOINT),
            "expressDelete" =>
                WC_AJAX::get_endpoint(ExpressDeleteEndpoint::ENDPOINT) .
                "&nonce=" .
                wp_create_nonce(ExpressDeleteEndpoint::ENDPOINT),
            "cardSubmitCallback" =>
                WC_AJAX::get_endpoint(CardSubmitCallbackEndpoint::ENDPOINT) .
                "&nonce=" .
                wp_create_nonce(CardSubmitCallbackEndpoint::ENDPOINT),
        ];
    }

    /**
     * Get the Rvvup Express Payment parameters for the rvvup_parameters js object.
     *
     * @return array
     */
    private static function getExpressPaymentParameters(): array
    {
        $rvvupExpressParameters = [
            "isExpress" => 0,
            "id" => 0,
            "cancel_url" => "",
        ];

        // Set params if Express Payment is used in the session.
        $rvvupExpressOrder = SessionManager::getInstance()->getRvvupExpressOrder();

        if (!is_array($rvvupExpressOrder)) {
            return $rvvupExpressParameters;
        }

        $rvvupExpressParameters["isExpress"] = isset($rvvupExpressOrder["id"]) ? 1 : 0;
        $rvvupExpressParameters["id"] = $rvvupExpressOrder["id"] ?? 0;
        $rvvupExpressParameters["cancel_url"] = $rvvupExpressOrder["cancel_url"] ?? "";

        return $rvvupExpressParameters;
    }

    /**
     * Get the Rvvup available payment method names for the rvvup_parameters js object.
     *
     * @param array $methods
     * @return array
     */
    private static function getRvvupPaymentMethodsParameters(array $methods): array
    {
        $rvvupPaymentMethods = [];

        foreach ($methods as $method) {
            if (!isset($method["name"])) {
                continue;
            }

            $rvvupPaymentMethods[] = [
                "id" => "rvvup_gateway_" . $method["name"],
                "name" => $method["name"],
                "displayName" => $method["displayName"],
                "icon" => $method["logoUrl"],
                "summaryUrl" => $method["summaryUrl"],
                "description" => $method["description"],
                "limits" => $method["limits"] ?? [],
            ];
        }

        return $rvvupPaymentMethods;
    }

    /**
     * Get the Rvvup available payment method settings for the rvvup_parameters js object.
     *
     * Remove assets from settings as they are already loaded.
     *
     * @param array $methods
     * @return array
     */
    private static function getRvvupPaymentMethodsSettings(array $methods): array
    {
        $rvvupPaymentMethodSettings = [];

        foreach ($methods as $method) {
            if (!isset($method["name"], $method["settings"]) || !is_array($method["settings"])) {
                continue;
            }

            $lowerCaseMethodName = mb_strtolower($method["name"]);

            $rvvupPaymentMethodSettings[$lowerCaseMethodName] = $method["settings"];

            if (isset($rvvupPaymentMethodSettings[$lowerCaseMethodName]["assets"])) {
                unset($rvvupPaymentMethodSettings[$lowerCaseMethodName]["assets"]);
            }
        }

        return $rvvupPaymentMethodSettings;
    }

    /**
     * Get various Site URLs that are useful for frontend.
     *
     * @return string[]
     */
    private static function getSiteUrls()
    {
        return [
            "checkout_url" => function_exists("wc_get_checkout_url") ? wc_get_checkout_url() : "",
        ];
    }
}
