<?php declare(strict_types=1);

namespace Rvvup\Payments\Model;

/**
 * Allow updating of the plugin via the WordPress admin.
 * Modified version of https://github.com/rudrastyh/misha-update-checker
 */
class ClosedBetaUpdater
{
    /** @var string */
    public $pluginSlug;
    /** @var string */
    public $version;
    /** @var string */
    public $cacheKey;
    /** @var bool */
    public $cacheAllowed = true;

    public function __construct()
    {
        $this->pluginSlug = plugin_basename(RVVUP_PLUGIN_FILE);
        $this->version = get_file_data(RVVUP_PLUGIN_FILE, ["Version" => "Version"], "plugin")["Version"];
        $this->cacheKey = "rvvup-plugin-updates";
        add_filter("plugins_api", [$this, "info"], 20, 3);
        add_filter("site_transient_update_plugins", [$this, "update"]);
        add_action("upgrader_process_complete", [$this, "purge"], 10, 2);
    }

    public function request()
    {
        $remote = get_transient($this->cacheKey);
        if (false === $remote || !$this->cacheAllowed) {
            $remote = wp_remote_get("https://assets.rvvup.com/plugins/woocommerce/releases/release-manifest.json", [
                "timeout" => 10,
                "headers" => [
                    "Accept" => "application/json",
                ],
            ]);

            if (
                is_wp_error($remote) ||
                200 !== wp_remote_retrieve_response_code($remote) ||
                empty(wp_remote_retrieve_body($remote))
            ) {
                return false;
            }
            set_transient($this->cacheKey, $remote, HOUR_IN_SECONDS);
        }
        return json_decode(wp_remote_retrieve_body($remote), true);
    }

    function info($res, $action, $args)
    {
        // do nothing if you're not getting plugin information right now
        if ("plugin_information" !== $action) {
            return $res;
        }
        // do nothing if it is not our plugin
        if ($this->pluginSlug !== $args->slug) {
            return $res;
        }

        // get updates
        $remote = $this->request();

        if (!$remote) {
            return $res;
        }
        if (is_array($remote)) {
            // Object casting is required in order to match the info/1.0 format.
            return (object) $remote;
        }

        return $remote;
    }

    public function update($transient)
    {
        if (empty($transient->checked)) {
            return $transient;
        }
        $remote = $this->request();
        if (
            $remote &&
            $remote["version"] &&
            $remote["download_link"] &&
            $remote["requires"] &&
            $remote["requires_php"] &&
            get_bloginfo("version") &&
            version_compare($this->version, $remote["version"], "!=") &&
            version_compare($remote["requires"], get_bloginfo("version"), "<=") &&
            version_compare($remote["requires_php"], PHP_VERSION, "<=")
        ) {
            $res = new \stdClass();
            $res->slug = $this->pluginSlug;
            $res->plugin = plugin_basename(RVVUP_PLUGIN_FILE);
            $res->new_version = $remote["version"];
            $res->tested = $remote["tested"];
            $res->package = $remote["download_link"];
            $transient->response[$res->plugin] = $res;
        }
        return $transient;
    }

    public function purge($unused, $options)
    {
        if ($this->cacheAllowed && "update" === $options["action"] && "plugin" === $options["type"]) {
            // just clean the cache when new plugin version is installed
            delete_transient($this->cacheKey);
        }
    }
}
