<?php

declare(strict_types=1);

namespace Rvvup\Payments\Model\DataStores;

use Abstract_WC_Order_Data_Store_CPT;

/**
 * RvvupRefundDataStore class that follows same practice & naming conventions as core WooCommerce Refund Data Store.
 */
class RvvupRefundDataStore extends Abstract_WC_Order_Data_Store_CPT
{
    /**
     * Data stored in meta keys, but not considered "meta" for an order.
     *
     * @var array
     */
    protected $internal_meta_keys = [
        "_refund_id",
        "_woocommerce_refund_id",
        "_refund_amount",
        "_refunded_by",
        "_refund_reason",
        "_is_full_refund",
    ];

    /**
     * Delete a refund - no trash is supported.
     *
     * @param \Rvvup\Payments\Model\PostTypes\RvvupRefund $order Order object.
     * @param array $args Array of args to pass to the delete method.
     *
     * @return void
     */
    public function delete(&$order, $args = [])
    {
        // Deleting Rvvup refunds is not supported.
    }

    /**
     * Read refund data. Can be overridden by child classes to load other props.
     *
     * @param \Rvvup\Payments\Model\PostTypes\RvvupRefund $refund
     * @param object $post_object Post object.
     */
    protected function read_order_data(&$refund, $post_object)
    {
        parent::read_order_data($refund, $post_object);

        $id = $refund->get_id();
        $refund->set_props([
            "refund_id" => metadata_exists("post", $id, "_refund_id") ? get_post_meta($id, "_refund_id", true) : "",
            "woocommerce_refund_id" => metadata_exists("post", $id, "_woocommerce_refund_id")
                ? get_post_meta($id, "_woocommerce_refund_id", true)
                : 0,
            "amount" => get_post_meta($id, "_refund_amount", true),
            "refunded_by" => metadata_exists("post", $id, "_refunded_by")
                ? get_post_meta($id, "_refunded_by", true)
                : absint($post_object->post_author),
            "reason" => metadata_exists("post", $id, "_refund_reason")
                ? get_post_meta($id, "_refund_reason", true)
                : $post_object->post_excerpt,
            "is_full_refund" =>
                metadata_exists("post", $id, "_is_full_refund") && get_post_meta($id, "_is_full_refund", true)
                    ? (int) wc_string_to_bool(get_post_meta($id, "_is_full_refund", true))
                    : 0,
        ]);
    }

    /**
     * Helper method that updates all the post meta for a Rvvup Refund based on its settings in the RvvupRefund class.
     *
     * @param \Rvvup\Payments\Model\PostTypes\RvvupRefund $refund
     * @since 3.0.0
     */
    protected function update_post_meta(&$refund)
    {
        parent::update_post_meta($refund);

        $meta_key_to_props = [
            "_refund_id" => "refund_id",
            "_woocommerce_refund_id" => "woocommerce_refund_id",
            "_refund_amount" => "amount",
            "_refunded_by" => "refunded_by",
            "_refund_reason" => "reason",
            "_is_full_refund" => "is_full_refund",
        ];

        $props_to_update = $this->get_props_to_update($refund, $meta_key_to_props);

        foreach ($props_to_update as $meta_key => $prop) {
            $value = $refund->{"get_$prop"}("edit");
            update_post_meta($refund->get_id(), $meta_key, $value);
        }
    }

    /**
     * Get a title for the new post type.
     *
     * @return string
     */
    protected function get_post_title()
    {
        return sprintf(
            __("Rvvup Refund &ndash; %s", "rvvup-for-woocommerce"),
            strftime(_x("%b %d, %Y @ %I:%M %p", "Order date parsed by strftime", "rvvup-for-woocommerce"))
        );
    }
}
