<?php

declare(strict_types=1);

namespace Rvvup\Payments\Model\PostTypes;

use WC_Abstract_Order;

/**
 * RvvupRefund class that follows same practice & naming conventions as core WooCommerce Refund Data Store.
 */
class RvvupRefund extends WC_Abstract_Order
{
    /**
     * Class constants
     */
    public const TYPE = "rvvup_refund";

    public const DATA_STORE_NAME = "rvvup-refund";

    /**
     * Which data store to load.
     *
     * @var string
     */
    protected $data_store_name = self::DATA_STORE_NAME;

    /**
     * This is the name of this object type.
     *
     * @var string
     */
    protected $object_type = self::TYPE;

    /**
     * Stores product data.
     *
     * @var array
     */
    protected $extra_data = [
        "refund_id" => null,
        "woocommerce_refund_id" => null,
        "amount" => "",
        "reason" => "",
        "refunded_by" => 0,
        "is_full_refund" => 0,
    ];

    /**
     * Get internal type (post type.)
     *
     * @return string
     */
    public function get_type(): string
    {
        return self::TYPE;
    }

    /**
     * Set the Rvvup Refund ID as returned via the API.
     *
     * @param $value
     * @return void
     */
    public function set_refund_id($value): void
    {
        $this->set_prop("refund_id", (string) $value);
    }

    /**
     * Get the Rvvup Refund's ID as it was set via the API.
     *
     * @param string $context
     * @return string
     */
    public function get_refund_id(string $context = "view"): string
    {
        return $this->get_prop("refund_id", $context);
    }

    /**
     * Set the original Woocommerce Refund ID (it will be deleted but this is required to avoid performing refund)
     *
     * @param int $value
     * @return void
     */
    public function set_woocommerce_refund_id($value): void
    {
        $this->set_prop("woocommerce_refund_id", absint($value));
    }

    /**
     * @param string $context
     * @return int
     */
    public function get_woocommerce_refund_id(string $context = "view"): int
    {
        return $this->get_prop("woocommerce_refund_id", $context);
    }

    /**
     * Set refunded amount.
     *
     * @param string $value Value to set.
     * @return void
     */
    public function set_amount($value): void
    {
        $this->set_prop("amount", wc_format_decimal($value));
    }

    /**
     * Get refunded amount.
     *
     * @param string $context What the value is for. Valid values are view and edit.
     * @return int|float
     */
    public function get_amount(string $context = "view"): string
    {
        return $this->get_prop("amount", $context);
    }

    /**
     * Set refund reason.
     *
     * @param string $value Value to set.
     * @return void
     */
    public function set_reason($value): void
    {
        $this->set_prop("reason", $value);
    }

    /**
     * Get refund reason.
     *
     * @param string $context What the value is for. Valid values are view and edit.
     * @return string
     */
    public function get_reason(string $context = "view"): string
    {
        return $this->get_prop("reason", $context);
    }

    /**
     * Set refunded by.
     *
     * @param int $value Value to set.
     * @return void
     */
    public function set_refunded_by($value): void
    {
        $this->set_prop("refunded_by", absint($value));
    }

    /**
     * Get ID of user who did the refund.
     *
     * @param string $context What the value is for. Valid values are view and edit.
     * @return int|null
     */
    public function get_refunded_by(string $context = "view"): ?int
    {
        return $this->get_prop("refunded_by", $context);
    }

    /**
     * Set the full refund flag.
     *
     * @param mixed $value
     * @return void
     */
    public function set_is_full_refund($value): void
    {
        $this->set_prop("is_full_refund", absint(wc_string_to_bool($value)));
    }

    /**
     * Get whether a refund is associated with a full refund, we expect values of 0/1 for boolean handling.
     *
     * @param string $context What the value is for. Valid values are view and edit.
     * @return int
     */
    public function get_is_full_refund(string $context = "view"): int
    {
        return (int) wc_string_to_bool($this->get_prop("is_full_refund", $context));
    }

    /**
     * Rvvup Refunds should always be created via a Rvvup API call.
     *
     * @param string $context
     * @return string
     */
    public function get_created_via(string $context = "view"): string
    {
        return "rvvup-api";
    }

    /**
     * Get a title for the new post type.
     *
     * @return string
     */
    public function get_post_title(): string
    {
        return sprintf(
            __("Rvvup Refund &ndash; %s", "rvvup-for-woocommerce"),
            strftime(_x("%b %d, %Y @ %I:%M %p", "Order date parsed by strftime", "rvvup-for-woocommerce"))
        );
    }

    /**
     * Get formatted refunded amount.
     *
     * @return string
     */
    public function getFormattedRvvupRefundAmount(): string
    {
        return apply_filters(
            "rvvup_formatted_rvvup_refund_amount",
            wc_price($this->get_amount(), ["currency" => $this->get_currency()]),
            $this
        );
    }
}
