<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Rvvup\Payments\Traits\GatewayTrait;
use WC_Order_Refund;

class AdminViewManager
{
    use GatewayTrait;

    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * @return void
     */
    public function init(): void
    {
        if (!is_admin()) {
            return;
        }

        add_action("woocommerce_after_order_refund_item_name", [$this, "renderRvvupRefundInOrderViewRefundLineItems"]);
        add_action("woocommerce_admin_order_totals_after_refunded", [
            $this,
            "renderRvvupRefundPendingTotalInOrderView",
        ]);
    }

    /**
     * Render Rvvup Refund details in the admin order view within the refund line items.
     *
     * @param \WC_Order_Refund $refund
     * @return void
     */
    public function renderRvvupRefundInOrderViewRefundLineItems(WC_Order_Refund $refund): void
    {
        if (!is_admin()) {
            return;
        }

        $rvvupRefund = RvvupRefundRepository::getInstance()->getByWooCommerceRefund($refund);

        if ($rvvupRefund === null) {
            return;
        }

        printf(
            '<p id="rvvup-refund-%1$s" class="rvvup-refund-item" data-rvvup-refund-id="%1$s">%2$s</p>',
            esc_html($rvvupRefund->get_id()),
            sprintf(
                '<p class="rvvup-refund-details">' .
                    esc_html__('Rvvup Refund #%1$s - %2$s - Status: %3$s - %4$s', "rvvup-for-woocommerce") .
                    "</p>",
                esc_html($rvvupRefund->get_id()),
                esc_html($rvvupRefund->get_refund_id()),
                esc_html(ucfirst($rvvupRefund->get_status())),
                esc_html(
                    wc_format_datetime(
                        $rvvupRefund->get_date_created(),
                        get_option("date_format") . ", " . get_option("time_format")
                    )
                )
            )
        );
    }

    /**
     * Render the total of Rvvup Refunds that wait confirmation in the order view page
     *
     * @param int $orderId
     * @return void
     */
    public function renderRvvupRefundPendingTotalInOrderView(int $orderId): void
    {
        if (!is_admin()) {
            return;
        }

        $order = wc_get_order($orderId);

        // Fail-safe, should not happen
        if (!$order) {
            return;
        }

        $total = RvvupRefundRepository::getInstance()->getPendingTotalByOrderId($orderId);

        if ($total === 0.0) {
            return;
        }

        printf(
            '<tr>%1$s%2$s%3$s</tr>',
            '<td class="label refunded-total">' .
                esc_html__("Pending Confirmation", "rvvup-for-woocommerce") .
                ":</td>",
            '<td width="1%"></td>',
            sprintf(
                '<td class="total refunded-total">%1$s</td>',
                wc_price($total, ["currency" => $order->get_currency()])
            )
        );
    }
}
