<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

class CheckoutPopulate
{
    private const BILLING = "billing_";
    private const SHIPPING = "shipping_";
    private const ADDRESS = "address";

    /**
     * @param $default
     * @param $fieldId
     * @return string|null
     */
    public static function rvvup_populate_checkout($default, $fieldId): ?string
    {
        $rvvupOrder = SessionManager::getInstance()->getRvvupExpressOrder();

        if (!$rvvupOrder) {
            return $default;
        }

        $needle = self::getNeedle($fieldId);

        if (!$needle) {
            return $default;
        }

        $address = $needle . self::ADDRESS;
        if (!isset($rvvupOrder[$address])) {
            return $default;
        }

        $fieldName = self::getFieldName($needle, $fieldId);
        $data = self::getRvvupAddressValue($fieldName, $address, $rvvupOrder);
        if ($data) {
            return $data;
        }

        return $default;
    }

    /**
     * Get field name by mappings
     * @param string $needle
     * @param string $fieldId
     * @return array|string|string[]
     */
    private static function getFieldName(string $needle, string $fieldId)
    {
        $keys = [
            "country" => "country_code",
            "address_1" => "address_line_1",
            "address_2" => "address_line_2",
            "phone" => "phone_number",
            "email" => "email_address",
            "postcode" => "post_code",
        ];

        $fieldName = str_replace($needle, "", $fieldId);
        return isset($keys[$fieldName]) ? $keys[$fieldName] : $fieldName;
    }

    /**
     * Get value from Rvvup address,
     * if it doesn't exist we take them from shipping
     * @param string $fieldName
     * @param string $address
     * @param array $rvvupOrder
     * @return string|null
     */
    private static function getRvvupAddressValue(string $fieldName, string $address, array $rvvupOrder): ?string
    {
        if (isset($rvvupOrder[$address][$fieldName]) && !empty($rvvupOrder[$address][$fieldName])) {
            return $rvvupOrder[$address][$fieldName];
        } else {
            if (
                isset($rvvupOrder[self::SHIPPING . self::ADDRESS][$fieldName]) &&
                !empty($rvvupOrder[self::SHIPPING . self::ADDRESS][$fieldName])
            ) {
                return $rvvupOrder[self::SHIPPING . self::ADDRESS][$fieldName];
            }
        }
        return null;
    }

    /**
     * Get needle by address type
     * @param string $field_id
     * @return string|null
     */
    private static function getNeedle(string $field_id): ?string
    {
        if (strpos($field_id, self::BILLING) === 0) {
            return self::BILLING;
        } elseif (strpos($field_id, self::SHIPPING) === 0) {
            return self::SHIPPING;
        }
        return null;
    }
}
