<?php declare(strict_types=1);

namespace Rvvup\Payments\Service;

use Exception;
use WC_Product;
use WC_Product_Variation;

class ClearpayMessaging
{
    /** @var bool|null */
    private static $clearpayIsEnabled = null;

    /**
     * Add the Clearpay logo to the product page if the item price is within max/min threshold values
     * @param $html
     * @return mixed|string
     * @throws Exception
     */
    public static function addClearpayLogoPdp($html)
    {
        if (!self::shouldShowMessaging("product")) {
            return $html;
        }

        global $product;
        if (!($product instanceof WC_Product) || !is_product()) {
            return $html;
        }

        // If item product type is not enabled for Rvvup, no message.
        // As this is the first instance we load Clearpay messaging on PDP,
        // we do not want this displayed for product variations.
        if (!RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductType($product->get_type(), false)) {
            return $html;
        }

        // If item product is restricted for CLEARPAY, no message.
        if (
            !RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductId(
                (string) $product->get_id(),
                "CLEARPAY"
            )
        ) {
            return $html;
        }

        $price = self::getProductPrice($product);
        if (!self::isInThreshold($price)) {
            return $html;
        }

        // Allows for correct refreshing when changing variant.
        // ClearPay messaging is added on `filterVariantPrices` method.
        if ($product->is_type("variable")) {
            return $html;
        }

        return $html . self::getMessagingHtml($price, "product");
    }

    public static function filterVariantPrices($prices, $product)
    {
        if (!self::shouldShowMessaging("product")) {
            return $prices;
        }

        // If not a product variation, no action.
        if (!$product instanceof WC_Product_Variation) {
            return $prices;
        }

        if (!RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductType($product->get_type())) {
            return $prices;
        }

        // If item product is restricted for CLEARPAY, no message.
        if (
            !RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductId(
                (string) $product->get_id(),
                "CLEARPAY"
            )
        ) {
            return $prices;
        }

        $price = self::getProductPrice($product);
        if (!self::isInThreshold($price)) {
            return $prices;
        }

        $prices .= self::getMessagingHtml($price, "product");

        return $prices;
    }

    /**
     * Add the Clearpay logo to the cart page if the item price is within max/min threshold values
     */
    public static function addClearpayLogoCart()
    {
        if (!self::shouldShowMessaging("cart")) {
            return;
        }

        if (!self::isInThreshold((string) WC()->cart->get_total(null))) {
            return;
        }

        $rvvupAvailableService = RvvupAvailable::getInstance();

        foreach (WC()->cart->get_cart() as $item) {
            // If item product type is not enabled for Rvvup, no message.
            if (!$rvvupAvailableService->isRvvupAvailableForProductByProductType($item["data"]->get_type())) {
                return;
            }

            // If item product is restricted for CLEARPAY, no message.
            if (
                !$rvvupAvailableService->isRvvupAvailableForProductByProductId(
                    (string) $item["data"]->get_id(),
                    "CLEARPAY"
                )
            ) {
                return;
            }
        }

        echo self::getMessagingHtml(WC()->cart->get_total(null), "cart");
    }

    /**
     * Determine if Clearpay messaging should be shown, factoring in WooCommerce settings and Rvvup settings
     * @param string $area
     * @return bool
     */
    private static function shouldShowMessaging(string $area): bool
    {
        try {
            if (!GatewaySettingsManager::getInstance()->isEnabled()) {
                return false;
            }

            return GatewaySettingsManager::getInstance()->shouldShowClearPayMessaging($area);
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Should Clearpay messaging and iconography be show for specified product?
     * @param $price
     * @return bool
     */
    private static function isInThreshold($price): bool
    {
        return ClearpayThresholdManager::getInstance()->get((string) $price, get_woocommerce_currency());
    }

    /**
     * Render messaging output
     * @param $amount
     * @param string $area
     * @return string
     */
    private static function getMessagingHtml($amount, string $area)
    {
        $currency = get_woocommerce_currency();
        $theme = GatewaySettingsManager::getInstance()->getClearPayTheme($area);
        return <<<HTML
<afterpay-placement
    data-locale="en_GB"
    data-currency="$currency"
    data-amount="$amount"
    data-badge-theme="$theme"
 ></afterpay-placement>
HTML;
    }

    /**
     * @param $product
     * @return float|null
     */
    private static function getProductPrice($product): ?float
    {
        $display_prices_in_shop = get_option("woocommerce_tax_display_shop");
        if ($display_prices_in_shop == "incl") {
            return wc_get_price_including_tax($product);
        }
        return wc_get_price_excluding_tax($product);
    }
}
