<?php declare(strict_types=1);

namespace Rvvup\Payments\Service;

use Automattic\WooCommerce\Admin\Overrides\Order;
use Exception;
use Rvvup\Payments\Gateway\Dynamic;
use Rvvup\Payments\Gateway\Fallback;
use Rvvup\Payments\Gateway\Rvvup;
use Rvvup\Payments\Traits\GatewayTrait;

class GatewayLoader
{
    use GatewayTrait;

    /**
     * ToDo: Investigate if this is still needed if method is removed.
     *
     * @var bool
     */
    private $onlyLoadDefault = false;
    /** @var bool */
    private $isCallback = false;
    /** @var string */
    private $currentMethod = "";
    /** @var string */
    private $currentTitle = "Rvvup";
    /** @var bool */
    private $hasRun = false;
    /** @var array */
    private $methods;
    /** @var array */
    private $sortOrder;

    /**
     * Register hooks
     */
    public function __construct()
    {
        // Main method to determine which gateways to load
        add_filter("woocommerce_payment_gateways", [$this, "loadGateways"]);
        add_filter("woocommerce_order_get_payment_method", [$this, "loadUsedMethod"]);
        add_action("woocommerce_create_refund", [$this, "loadSpecificMethod"]);
        // Ensure the order of gateways displayed matches the order in the Rvvup dashboard correctly
        add_filter("option_woocommerce_gateway_order", [$this, "reorderGateways"]);
        // When the WC gateway admin is loaded we only want to load the "parent" to allow merchant configuration
        add_action("load-woocommerce_page_wc-settings", [$this, "loadDefaultGateway"], 0);
    }

    /**
     * Set default gateway loading to true.
     * ToDo: Investigate if this is still needed.
     *
     * @return void
     */
    public function loadDefaultGateway()
    {
        $this->onlyLoadDefault = true;
    }

    /**
     * The filter callback function
     *
     * @param array $gateways
     * @return array
     */
    public static function loadInitialSetupGateway($gateways)
    {
        if (is_admin()) {
            $gateways[] = Rvvup::class;
        }
        return $gateways;
    }

    /**
     * The filter callback function.
     *
     * @param array $gateways
     * @return array
     */
    public function loadGateways($gateways)
    {
        /**
         * Scenarios where we only want to show the default option:
         * - WooCommerce payment gateways grid,
         */
        if ($this->onlyLoadDefault()) {
            $gateways[] = Rvvup::class;

            return $gateways;
        }

        /**
         * Scenarios where we want to load just the specific gateway:
         * - Order view screen,
         * - Refund creation,
         * - Webhooks,
         */
        if ($this->loadSpecific()) {
            $gateways[] = new Dynamic($this->currentMethod, $this->currentTitle, "", "", [], "");

            return $gateways;
        }

        /**
         * Scenarios where we want to load all available payment methods:
         * - Checkout. A cart is not always available. Determine a "total" regardless
         */
        $cart = WC()->cart;
        $total = $cart !== null ? $cart->get_total(null) : "0";
        try {
            $methods = SdkProxy::getMethods((string) $total);
            $country = $this->getCountry();
            if (!$this->isNullOrEmptyString($country) && $country !== "GB") {
                $this->removePaymentMethodByCode("YAPILY", $methods);
                $this->removePaymentMethodByCode("PAY_BY_BANK", $methods);
            }
            $this->methods = $methods;
        } catch (Exception $e) {
            $methods = [];
        }
        // Instantiate remaining gateways after filtering by restrictions
        foreach ($methods as $method) {
            if (!isset($method["name"])) {
                continue;
            }

            $gateways[] = new Dynamic(
                $method["name"],
                $method["displayName"] ?? "",
                getCheckoutIconPath($method["name"], $method["logoUrl"]),
                $method["summaryUrl"] ?? "",
                $method["settings"] ?? [],
                $method["captureType"] ?? ""
            );
        }
        if (!count($methods)) {
            $gateways[] = Fallback::class;
        }
        return $gateways;
    }

    /**
     * @param string|null $str
     * @return bool
     */
    private function isNullOrEmptyString(?string $str): bool
    {
        return $str === null || trim($str) === "";
    }

    /**
     * @param string $code
     * @param array $methods
     * @return void
     */
    private function removePaymentMethodByCode(string $code, array &$methods)
    {
        foreach ($methods as $key => $method) {
            if ($method["name"] === $code) {
                unset($methods[$key]);
                break;
            }
        }
    }

    /**
     * @return string|null
     */
    private function getCountry(): ?string
    {
        $country = null;
        if (is_checkout_pay_page()) {
            global $wp;
            $order_id = $wp->query_vars["order-pay"] ?? null;
            if ($order_id) {
                $order = wc_get_order($order_id) ?? null;
                if ($order) {
                    $country = $order->get_shipping_country() ?: $order->get_billing_country();
                }
            }
        }
        if (!$country) {
            if (isset($_POST["s_country"]) && $_POST["s_country"]) {
                $country = $_POST["s_country"];
            } else {
                $country = WC()->customer
                    ? (WC()->customer->get_shipping_country() ?:
                    WC()->customer->get_billing_country())
                    : null;
            }
        }
        return $country;
    }

    /**
     * @param $value
     * @return mixed
     */
    public function loadUsedMethod($value)
    {
        if ($this->hasRun) {
            return $value;
        }
        if ($this->isRvvupGateway($value)) {
            $this->isCallback = true;
            $this->currentMethod = $this->getRvvupMethodName($value);
            global $theorder;
            if ($theorder) {
                $this->currentTitle = $theorder->get_meta("_rvvup_title");
            }
            $this->hasRun = true;
            // re-load gateways
            WC()
                ->payment_gateways()
                ->init();
        }
        return $value;
    }

    /**
     * @param $refund
     * @return void
     */
    public function loadSpecificMethod($refund)
    {
        $order = wc_get_order($refund->get_parent_id());
        $method = $order->get_payment_method();
        $code = str_replace("rvvup_gateway_", "", $method);
        $this->isCallback = true;
        $this->currentMethod = $code;
        WC()
            ->payment_gateways()
            ->init();
    }

    /**
     * Ensure that we set the correct sort-ordering of payment gateways based on the number of dynamic methods created
     *
     * @param array $sortOrder
     * @return array
     */
    public function reorderGateways(array $sortOrder): array
    {
        if (!(array_key_exists("rvvup_gateway", $sortOrder) && $this->methods)) {
            return $sortOrder;
        }
        if ($this->sortOrder) {
            return $this->sortOrder;
        }
        $rvvupIndex = (int) $sortOrder["rvvup_gateway"];
        $rvvupMethods = [];
        foreach ($this->methods as $method) {
            $methodName = "rvvup_gateway_" . $method["name"];
            $rvvupMethods[$methodName] = null;
        }

        // Remove "placeholder" gateway element and replace with all Rvvup methods
        $sortOrder =
            array_slice($sortOrder, 0, $rvvupIndex, true) +
            $rvvupMethods +
            array_slice($sortOrder, $rvvupIndex, null, true);

        // Re-index sort-order positions
        $index = 0;
        foreach ($sortOrder as $key => $position) {
            $sortOrder[$key] = $index++;
        }
        $this->sortOrder = $sortOrder;
        return $this->sortOrder;
    }

    /**
     * @return bool
     */
    private function loadSpecific(): bool
    {
        global $theorder;

        if ($theorder instanceof Order) {
            $this->isCallback = true;
            $this->currentMethod = str_replace("rvvup_gateway_", "", $theorder->get_payment_method());
            $this->currentTitle = $theorder->get_payment_method_title();

            return true;
        }
        return false;
    }

    /**
     * @return bool
     */
    private function onlyLoadDefault(): bool
    {
        /**
         * If the gateway is disabled we only want to load the default to allow an admin usr to change the settings
         */
        if (!wc_string_to_bool((new Rvvup())->settings["enabled"])) {
            return true;
        }
        /**
         * When saving the gateway setting for Rvvup there appears to be some REST API related code interfering with
         * our gateway loader. This allows us to identify when this is the page being loaded and ensure only the "Base"
         * gateway is loaded
         */
        if (
            isset($_GET["page"], $_GET["tab"]) &&
            $_GET["page"] === "wc-settings" &&
            $_GET["tab"] === "checkout" &&
            is_admin()
        ) {
            return true;
        }
        // Ensure that base gateway is loaded when toggling Rvvup on/off
        if (
            isset($_POST["action"], $_POST["gateway_id"]) &&
            $_POST["action"] === "woocommerce_toggle_gateway_enabled" &&
            $_POST["gateway_id"] === "rvvup_gateway"
        ) {
            return true;
        }
        // ToDo: investigate if that's still needed or we can just return false.
        return $this->onlyLoadDefault;
    }
}
