<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

use Rvvup\Payments\Contract\Service\GatewaySettingsManagerInterface;
use Rvvup\Payments\Gateway\Dynamic;
use Rvvup\Payments\Gateway\Rvvup;
use Rvvup\Payments\ViewModel\Clearpay;

/**
 * This class should be used when we want to retrieve Rvvup payment gateway settings/config if plugin is already set.
 * So, we assume the class can be instantiated.
 * For direct database config, use get_option("woocommerce_rvvup_gateway_settings", null) as per SdkProxy.
 */
class GatewaySettingsManager implements GatewaySettingsManagerInterface
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * @var array|null
     */
    private $settings;

    /**
     * @var array|null
     */
    private $rvvupSettings;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Check whether Rvvup is enabled.
     *
     * @return bool
     */
    public function isEnabled(): bool
    {
        $settings = $this->getSettings();

        return isset($settings["enabled"]) && wc_string_to_bool($this->getSettings()["enabled"]);
    }

    /**
     * Get the merchant ID.
     *
     * @return string|null
     */
    public function getMerchantId(): ?string
    {
        $settings = $this->getSettings();

        return isset($settings[SdkProxy::MERCHANT_ID]) && is_string($settings[SdkProxy::MERCHANT_ID])
            ? $settings[SdkProxy::MERCHANT_ID]
            : null;
    }

    /**
     * Get the enabled product types.
     *
     * @return array
     */
    public function getEnabledProductTypes(): array
    {
        $settings = $this->getSettings();

        return isset($settings["enabled_product_types"]) && is_array($settings["enabled_product_types"])
            ? $settings["enabled_product_types"]
            : [];
    }

    /**
     * Get the api data settings loaded in the Gateway.
     *
     * @param \Rvvup\Payments\Gateway\Dynamic $gateway
     * @param string $path
     * @return mixed|null
     */
    public function getApiConfig(Dynamic $gateway, string $path)
    {
        return $gateway->getApiConfig($path);
    }

    /**
     * Get the modal position for mobile layout.
     *
     * @return string|null
     */
    public function getMobileModalPosition(): ?string
    {
        $settings = $this->getSettings();

        return $settings["mobile_position"] ?? null;
    }

    /**
     * Check whether ClearPay Messaging should be displayed (enabled display).
     *
     * Defaults to true.
     *
     * @return bool
     */
    public function shouldShowClearPayMessaging(string $area): bool
    {
        $settings = $this->getRvvupSettings($area);
        return $settings && $settings["messaging"]["enabled"];
    }

    /**
     * Get ClearPay theme.
     *
     * Defaults to Black on Mint
     *
     * @param string $area
     * @return string
     */
    public function getClearPayTheme(string $area): string
    {
        $settings = $this->getRvvupSettings($area);
        return $settings ? (string) $settings["theme"]["value"] : "";
    }

    /**
     * Is Debug mode enabled?
     *
     * @return bool
     */
    public function isDebugEnabled(): bool
    {
        return isset($this->getSettings()["debug"]) && wc_string_to_bool($this->getSettings()["debug"]);
    }

    /**
     * Get the publishable key for js sdk.
     *
     * @return string|null
     */
    public function getPublishableKey(): ?string
    {
        return $this->getMerchantId();
    }

    /**
     * Get the JS SDK Url
     *
     * @return string|null
     */
    public function getJsSdkUrl(): ?string
    {
        $settings = $this->getSettings();

        if (!isset($settings[SdkProxy::BACKEND_API_URL]) || !is_string($settings[SdkProxy::BACKEND_API_URL])) {
            return null;
        }

        $url = str_replace("graphql", "sdk/v2.js", $settings[SdkProxy::BACKEND_API_URL]);
        return str_replace("api.", "checkout.", $url);
    }

    /**
     * Load & fetch Rvvup gateway settings.
     *
     * If not null, it will be an array as loaded via class instantiation.
     *
     * @return array
     */
    private function getSettings(): array
    {
        if ($this->settings !== null) {
            return $this->settings;
        }

        $this->settings = (new Rvvup())->settings;
        return $this->settings;
    }

    /**
     * @param string $area
     * @return array
     * @throws \Exception
     */
    private function getRvvupSettings(string $area): array
    {
        if (!isset($this->rvvupSettings[$area])) {
            foreach (SdkProxy::getMethods("0") as $method) {
                if ($method["name"] == "CLEARPAY") {
                    $result = $method["settings"][$area];
                    $this->rvvupSettings[$area] = $result;
                }
            }

            if (!isset($this->rvvupSettings[$area])) {
                $this->rvvupSettings[$area] = [];
            }
        }

        return $this->rvvupSettings[$area];
    }
}
