<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

use WC_Logger_Interface;

class LoggerManager
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * @var WC_Logger_Interface|null
     */
    private $wcLogger;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function debug(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->debug(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function info(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->info(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function notice(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->notice(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function warning(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->warning(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function error(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->error(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function critical(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->critical(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function alert(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->alert(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * @param string $message
     * @param array $metadata
     * @param array $context
     * @return void
     */
    public function emergency(string $message, array $metadata = [], array $context = []): void
    {
        if ($this->getWcLogger() === null) {
            return;
        }

        $this->getWcLogger()->emergency(json_encode($metadata) . " | " . $message, $this->addSourceToContext($context));
    }

    /**
     * Set & get WC_Logger instance if set
     *
     * @return \WC_Logger_Interface|null
     */
    private function getWcLogger(): ?WC_Logger_Interface
    {
        if ($this->wcLogger === null) {
            $this->wcLogger = wc_get_logger();
        }

        return $this->wcLogger;
    }

    /**
     * Add source to the context array (so we use a custom log file)
     *
     * @param array $context
     * @return array
     */
    private function addSourceToContext(array $context): array
    {
        return $context;

        // ToDo: Not currently in use. Use when all plugin logs are handled by LoggerManager.
        if (array_key_exists("source", $context)) {
            return $context;
        }

        return array_merge($context, ["source" => "rvvup-for-woocomerce"]);
    }
}
