<?php

namespace Rvvup\Payments\Service;

use WC_Order;

class OrderDetailService
{
    /**
     * @var OrderDetailService|null
     */
    private static $instance = null;

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct()
    {
    }

    /**
     * Get the singleton instance
     *
     * @return OrderDetailService
     */
    public static function getInstance(): OrderDetailService
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Sync order with details from rvvup order data (response in API).
     * This currently only impacts orders placed with Rvvup Zopa Retail Finance method, and other methods
     * are ignored.
     * @param WC_Order|null $order
     * @param array|null $rvvupData
     */
    public function syncOrderWithRvvupData($order, ?array $rvvupData)
    {
        if ($order == null || $order->get_payment_method() !== "rvvup_gateway_ZOPA_RETAIL_FINANCE") {
            return;
        }
        if (empty($rvvupData)) {
            return;
        }
        $changes = [];

        $customerFirstName = $rvvupData["customer"]["givenName"] ?? "";
        $customerSurname = $rvvupData["customer"]["surname"] ?? "";

        // Set billing address
        if (isset($rvvupData["billingAddress"])) {
            $nameParts = !empty($rvvupData["billingAddress"]["name"])
                ? explode(" ", $rvvupData["billingAddress"]["name"], 2)
                : [$customerFirstName, $customerSurname];

            // Track billing address changes
            $newFirstName = $nameParts[0];
            $newLastName = $nameParts[1] ?? "";
            if ($order->get_billing_first_name() !== $newFirstName) {
                $changes[] = [
                    "field" => "Billing Address First Name",
                    "from" => $order->get_billing_first_name(),
                    "to" => $newFirstName,
                ];
                $order->set_billing_first_name($newFirstName);
            }
            if ($order->get_billing_last_name() !== $newLastName) {
                $changes[] = [
                    "field" => "Billing Address Last Name",
                    "from" => $order->get_billing_last_name(),
                    "to" => $newLastName,
                ];
                $order->set_billing_last_name($newLastName);
            }

            if ($order->get_billing_address_1() !== $rvvupData["billingAddress"]["line1"]) {
                $changes[] = [
                    "field" => "Billing Address Street",
                    "from" => $order->get_billing_address_1(),
                    "to" => $rvvupData["billingAddress"]["line1"],
                ];
                $order->set_billing_address_1($rvvupData["billingAddress"]["line1"]);
            }
            if ($order->get_billing_address_2() !== $rvvupData["billingAddress"]["line2"]) {
                $changes[] = [
                    "field" => "Billing Address Line 2",
                    "from" => $order->get_billing_address_2(),
                    "to" => $rvvupData["billingAddress"]["line2"],
                ];
                $order->set_billing_address_2($rvvupData["billingAddress"]["line2"]);
            }

            if ($order->get_billing_postcode() !== $rvvupData["billingAddress"]["postcode"]) {
                $changes[] = [
                    "field" => "Billing Address Postcode",
                    "from" => $order->get_billing_postcode(),
                    "to" => $rvvupData["billingAddress"]["postcode"],
                ];
                $order->set_billing_postcode($rvvupData["billingAddress"]["postcode"]);
            }

            if ($order->get_billing_state() !== $rvvupData["billingAddress"]["state"]) {
                $changes[] = [
                    "field" => "Billing Address State",
                    "from" => $order->get_billing_state(),
                    "to" => $rvvupData["billingAddress"]["state"],
                ];
                $order->set_billing_state($rvvupData["billingAddress"]["state"]);
            }

            if ($order->get_billing_city() !== $rvvupData["billingAddress"]["city"]) {
                $changes[] = [
                    "field" => "Billing Address City",
                    "from" => $order->get_billing_city(),
                    "to" => $rvvupData["billingAddress"]["city"],
                ];
                $order->set_billing_city($rvvupData["billingAddress"]["city"]);
            }

            if ($order->get_billing_phone() !== $rvvupData["billingAddress"]["phoneNumber"]) {
                $changes[] = [
                    "field" => "Billing Address Phone",
                    "from" => $order->get_billing_phone(),
                    "to" => $rvvupData["billingAddress"]["phoneNumber"],
                ];
                $order->set_billing_phone($rvvupData["billingAddress"]["phoneNumber"]);
            }
            if ($order->get_billing_country() !== $rvvupData["billingAddress"]["countryCode"]) {
                $changes[] = [
                    "field" => "Billing Address Country",
                    "from" => $order->get_billing_country(),
                    "to" => $rvvupData["billingAddress"]["countryCode"],
                ];
                $order->set_billing_country($rvvupData["billingAddress"]["countryCode"]);
            }
        }

        if (isset($rvvupData["shippingAddress"])) {
            $nameParts = !empty($rvvupData["shippingAddress"]["name"])
                ? explode(" ", $rvvupData["shippingAddress"]["name"], 2)
                : [$customerFirstName, $customerSurname];

            // Track billing address changes
            $newFirstName = $nameParts[0];
            $newLastName = $nameParts[1] ?? "";
            if ($order->get_shipping_first_name() !== $newFirstName) {
                $changes[] = [
                    "field" => "Shipping Address First Name",
                    "from" => $order->get_shipping_first_name(),
                    "to" => $newFirstName,
                ];
                $order->set_shipping_first_name($newFirstName);
            }
            if ($order->get_shipping_last_name() !== $newLastName) {
                $changes[] = [
                    "field" => "Shipping Address Last Name",
                    "from" => $order->get_shipping_last_name(),
                    "to" => $newLastName,
                ];
                $order->set_shipping_last_name($newLastName);
            }

            if ($order->get_shipping_address_1() !== $rvvupData["shippingAddress"]["line1"]) {
                $changes[] = [
                    "field" => "Shipping Address Street",
                    "from" => $order->get_shipping_address_1(),
                    "to" => $rvvupData["shippingAddress"]["line1"],
                ];
                $order->set_shipping_address_1($newFirstName);
            }
            if ($order->get_shipping_address_2() !== $rvvupData["shippingAddress"]["line2"]) {
                $changes[] = [
                    "field" => "Shipping Address Line 2",
                    "from" => $order->get_shipping_address_2(),
                    "to" => $rvvupData["shippingAddress"]["line2"],
                ];
                $order->set_shipping_address_2($newFirstName);
            }

            if ($order->get_shipping_postcode() !== $rvvupData["shippingAddress"]["postcode"]) {
                $changes[] = [
                    "field" => "Shipping Address Postcode",
                    "from" => $order->get_shipping_postcode(),
                    "to" => $rvvupData["shippingAddress"]["postcode"],
                ];
                $order->set_shipping_postcode($rvvupData["shippingAddress"]["postcode"]);
            }

            if ($order->get_shipping_state() !== $rvvupData["shippingAddress"]["state"]) {
                $changes[] = [
                    "field" => "Shipping Address State",
                    "from" => $order->get_shipping_state(),
                    "to" => $rvvupData["shippingAddress"]["state"],
                ];
                $order->set_shipping_state($rvvupData["shippingAddress"]["state"]);
            }

            if ($order->get_shipping_city() !== $rvvupData["shippingAddress"]["city"]) {
                $changes[] = [
                    "field" => "Shipping Address City",
                    "from" => $order->get_shipping_city(),
                    "to" => $rvvupData["shippingAddress"]["city"],
                ];
                $order->set_shipping_city($rvvupData["shippingAddress"]["city"]);
            }

            // get_shipping_phone() is only supported on Woocommerce 5.6.0 or higher.
            if (method_exists($order, "get_shipping_phone")) {
                if ($order->get_shipping_phone() !== $rvvupData["shippingAddress"]["phoneNumber"]) {
                    $changes[] = [
                        "field" => "Shipping Address Phone",
                        "from" => $order->get_shipping_phone(),
                        "to" => $rvvupData["shippingAddress"]["phoneNumber"],
                    ];
                    $order->set_shipping_phone($rvvupData["shippingAddress"]["phoneNumber"]);
                }
            }
            if ($order->get_shipping_country() !== $rvvupData["shippingAddress"]["countryCode"]) {
                $changes[] = [
                    "field" => "Shipping Address Country",
                    "from" => $order->get_shipping_country(),
                    "to" => $rvvupData["shippingAddress"]["countryCode"],
                ];
                $order->set_shipping_country($rvvupData["shippingAddress"]["phoneNumber"]);
            }
        }

        // Set email
        $customerEmail = $rvvupData["customer"]["email"] ?? null;
        if (!empty($customerEmail)) {
            if ($order->get_billing_email() !== $customerEmail) {
                $changes[] = [
                    "field" => "Billing Email",
                    "from" => $order->get_billing_email(),
                    "to" => $customerEmail,
                ];
                $order->set_billing_email($customerEmail);
            }
        }

        if (!empty($changes)) {
            $message =
                "Order details have CHANGED because the customer changed them " .
                "during the DivideBuy Checkout Flow:<br /><br />" .
                implode(
                    "<br />",
                    array_map(function ($change) {
                        $change["from"] = $change["from"] === null ? "N/A" : $change["from"];
                        $change["to"] = $change["to"] === null ? "N/A" : $change["to"];
                        return "- <strong>" .
                            $change["field"] .
                            "</strong> changed from '<strong>" .
                            $change["from"] .
                            "</strong>' to '<strong>" .
                            $change["to"] .
                            "</strong>'";
                    }, $changes)
                );

            $order->add_order_note($message, 0);
            $order->save();
        }
    }
}
