<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

use Exception;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

class OrderStatusManager
{
    /**
     * Custom order statuses constants.
     */
    public const STATUS_REFUND_PENDING = "wc-refund-pending";
    public const STATUS_REFUND_FAILED = "wc-refund-failed";

    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     *
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * @return void
     */
    public function init(): void
    {
        add_filter("woocommerce_register_shop_order_post_statuses", [$this, "registerCustomOrderStatuses"]);
        add_filter("wc_order_statuses", [$this, "includeCustomOrderStatuses"]);
        add_filter("woocommerce_order_fully_refunded_status", [$this, "handleFullyRefundedOrderStatus"], 10, 3);
    }

    /**
     * Add custom order statuses to registered statuses.
     *
     * @param array $statuses
     * @return array
     */
    public function registerCustomOrderStatuses(array $statuses): array
    {
        return array_merge($statuses, $this->getCustomStatuses());
    }

    /**
     * Add custom statuses when a list of order statuses is requested.
     *
     * @param array $statuses
     * @return array
     */
    public function includeCustomOrderStatuses(array $statuses): array
    {
        $customStatuses = $this->getCustomStatuses();

        foreach ($customStatuses as $key => $value) {
            $statuses[$key] = $value["label"];
        }

        return $statuses;
    }

    /**
     * Set order status as Pending Refund on Full Refund action, if refund is against a Rvvup payment.
     *
     * The Rvvup API Refund call is executed just before this hook, failed should return errors
     * & success should allow to continue as normal.
     * We only want to change status for Rvvup Refunds that are in pending status.
     *
     * @param string $status
     * @param int $orderId
     * @param int $refundId
     * @return string
     */
    public function handleFullyRefundedOrderStatus(string $status, int $orderId, int $refundId): string
    {
        // We want to handle the core behaviour of refunded, if this changed from something else, do not handle.
        if ($status !== "refunded") {
            return $status;
        }

        try {
            $rvvupRefund = RvvupRefundRepository::getInstance()->getByOrderIdAndWooCommerceRefundId(
                $orderId,
                $refundId
            );

            // If the refund is not associated to a Rvvup Refund (meaning we did not refund a Rvvup payment)
            // or the rvvup refund status is not pending (only status we need to handle).
            // return as is.
            if ($rvvupRefund === null || !$rvvupRefund->has_status("pending")) {
                return $status;
            }

            return static::STATUS_REFUND_PENDING;
        } catch (Exception $ex) {
            $message = "Error thrown when searching for Rvvup Refunds by order & refund IDs. ";
            LoggerManager::getInstance()->error($message, [
                "order_id" => $orderId,
                "refund_id" => $refundId,
            ]);
            return $status;
        }
    }

    /**
     * Get the custom statuses.
     *
     * @return array[]
     */
    private function getCustomStatuses(): array
    {
        return [
            static::STATUS_REFUND_PENDING => [
                "label" => _x("Pending refund", "Order status", "rvvup-for-woocommerce"),
                "public" => false,
                "exclude_from_search" => false,
                "show_in_admin_all_list" => true,
                "show_in_admin_status_list" => true,
                "label_count" => _n_noop(
                    'Pending refund <span class="count">(%s)</span>',
                    'Pending refund <span class="count">(%s)</span>',
                    "rvvup-for-woocommerce"
                ),
            ],
            static::STATUS_REFUND_FAILED => [
                "label" => _x("Failed refund", "Order status", "rvvup-for-woocommerce"),
                "public" => false,
                "exclude_from_search" => false,
                "show_in_admin_all_list" => true,
                "show_in_admin_status_list" => true,
                "label_count" => _n_noop(
                    'Failed refund <span class="count">(%s)</span>',
                    'Failed refund <span class="count">(%s)</span>',
                    "rvvup-for-woocommerce"
                ),
            ],
        ];
    }
}
