<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Rvvup\Payments\Exceptions\InvalidActionException;

class RefundManager
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Intialize service & hooks.
     *
     * @return void
     */
    public function init(): void
    {
        add_filter("pre_delete_post", [$this, "preventDeletingWcRefund"], 10, 2);
    }

    /**
     * Disable all refund email notifications that would go out with current request.
     *
     * @return void
     */
    public function disableRefundNotifications(): void
    {
        $transactionalEmailActions = ["woocommerce_order_fully_refunded", "woocommerce_order_partially_refunded"];

        remove_all_actions("woocommerce_order_fully_refunded_notification");
        remove_all_actions("woocommerce_order_partially_refunded_notification");

        if (apply_filters("woocommerce_defer_transactional_emails", false)) {
            foreach ($transactionalEmailActions as $transactionalEmailAction) {
                remove_action($transactionalEmailAction, ["WC_Emails", "queue_transactional_email"]);
            }

            return;
        }

        foreach ($transactionalEmailActions as $transactionalEmailAction) {
            remove_action($transactionalEmailAction, ["WC_Emails", "send_transactional_email"]);
        }
    }

    /**
     * Trigger relevant notifications depending if Rvvup Refund was a full or partial refund.
     *
     * @param int $orderId
     * @param int $refundId
     * @param bool $full
     * @return void
     */
    public function triggerRefundNotifications(int $orderId, int $refundId, bool $full = false): void
    {
        if ($full) {
            do_action("woocommerce_order_fully_refunded", $orderId, $refundId);

            return;
        }

        do_action("woocommerce_order_partially_refunded", $orderId, $refundId);
    }

    /**
     * Prevent deleting a WooCommerce Refund, if there is an associated Rvvup Refund
     * which has status other than failed (`wc-failed`).
     *
     * @param mixed $check
     * @param mixed $post
     * @return mixed
     * @throws \Rvvup\Payments\Exceptions\InvalidActionException
     */
    public function preventDeletingWcRefund($check, $post)
    {
        if (get_post_type($post) !== "shop_order_refund") {
            return $check;
        }

        $refund = wc_get_order($post);

        if (!$refund) {
            return $check;
        }

        $rvvupRefund = RvvupRefundRepository::getInstance()->getByWooCommerceRefund($refund);

        // If there is no associated Rvvup Refund, or it is not in a pending state, then we can delete it
        if ($rvvupRefund === null || !$rvvupRefund->has_status("pending")) {
            return $check;
        }

        // Otherwise, throw exception as we should not be allowed to delete a pending Rvvup Refund
        throw new InvalidActionException("Not allowed to delete refund if associated Rvvup Refund is still pending.");
    }
}
