<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

class RvvupAvailable
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Check if Rvvup is enabled
     *
     * @return bool
     */
    public function isEnabled(): bool
    {
        return GatewaySettingsManager::getInstance()->isEnabled();
    }

    /**
     * If payment method is clearpay and product is Rvvup restricted, payment method is not available.
     *
     * @param string $productId
     * @param string $paymentMethodCode
     * @return bool
     */
    public function isRvvupAvailableForProductByProductId(string $productId, string $paymentMethodCode): bool
    {
        return !(
            (mb_strtolower($paymentMethodCode) === "clearpay" ||
                mb_strtolower($paymentMethodCode) === "zopa_retail_finance") &&
            wc_string_to_bool(get_post_meta($productId, "_rvvup_restricted", true))
        );
    }

    /**
     * If product type is not one of the enabled types, payment method is not available.
     * Also map the product type to the parent product's type if required from the area the method is called.
     *
     * @param string $productType
     * @param bool $mapToParentType
     * @return bool
     */
    public function isRvvupAvailableForProductByProductType(string $productType, bool $mapToParentType = true): bool
    {
        // Match a product type to the parent product's type if required.
        // Example during checkout or the variation product description HTML
        if ($mapToParentType && $productType === "variation") {
            $productType = "variable";
        }

        return in_array($productType, GatewaySettingsManager::getInstance()->getEnabledProductTypes(), true);
    }
}
