<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

class RvvupRefundManager
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Initialize hooks.
     *
     * @return void
     */
    public function init(): void
    {
        add_action("woocommerce_order_partially_refunded", [$this, "markAsPartialRefund"], 10, 2);
        add_action("woocommerce_order_fully_refunded", [$this, "markAsFullRefund"], 10, 2);
    }

    /**
     * Mark a RvvupRefund as associated to a partial refund.
     *
     * @param int $orderId
     * @param int $refundId
     * @return void
     */
    public function markAsPartialRefund(int $orderId, int $refundId): void
    {
        $rvvupRefund = RvvupRefundRepository::getInstance()->getByOrderIdAndWooCommerceRefundId($orderId, $refundId);

        if ($rvvupRefund === null) {
            return;
        }

        $rvvupRefund->set_is_full_refund(0); // Default value setter is false.
        $rvvupRefund->save();
    }

    /**
     * Mark a RvvupRefund as associated to a full refund.
     *
     * @param int $orderId
     * @param int $refundId
     * @return void
     */
    public function markAsFullRefund(int $orderId, int $refundId): void
    {
        $rvvupRefund = RvvupRefundRepository::getInstance()->getByOrderIdAndWooCommerceRefundId($orderId, $refundId);

        if ($rvvupRefund === null) {
            return;
        }

        $rvvupRefund->set_is_full_refund(1);
        $rvvupRefund->save();
    }
}
