<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Rvvup\Payments\Model\DataStores\RvvupRefundDataStore;
use Rvvup\Payments\Model\PostTypes\RvvupRefund;
use WC_Cache_Helper;
use WC_Order_Refund;

class RvvupRefundRepository
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * @var string
     */
    private $cacheGroup = "orders";

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Initialize
     *
     * @return void
     */
    public function init(): void
    {
        add_filter("woocommerce_data_stores", [$this, "installDataStore"]);
    }

    /**
     * Add up RvvupRefund Data Store to system stores.
     *
     * @param array $stores
     * @return array
     */
    public function installDataStore(array $stores = []): array
    {
        $stores[RvvupRefund::DATA_STORE_NAME] = new RvvupRefundDataStore();

        return $stores;
    }

    /**
     * Get the RvvupRefund entity by the linked refund.
     *
     * @param \WC_Order_Refund $refund
     * @return \Rvvup\Payments\Model\PostTypes\RvvupRefund|null
     */
    public function getByWooCommerceRefund(WC_Order_Refund $refund): ?RvvupRefund
    {
        return $this->getByOrderIdAndWooCommerceRefundId($refund->get_parent_id(), $refund->get_id());
    }

    /**
     * Get the RvvupRefund entity by the parent order ID & the linked refund id.
     *
     * @param int $orderId
     * @param int $refundId
     * @return \Rvvup\Payments\Model\PostTypes\RvvupRefund|null
     */
    public function getByOrderIdAndWooCommerceRefundId(int $orderId, int $refundId): ?RvvupRefund
    {
        $cacheKey =
            WC_Cache_Helper::get_cache_prefix($this->cacheGroup) .
            "rvvup-refund_order-id-" .
            $orderId .
            "_" .
            "wc-refund-id-" .
            $refundId;

        $cachedData = wp_cache_get($cacheKey, $this->cacheGroup);

        if ($cachedData !== false) {
            return $cachedData;
        }

        // Using meta_query array arguments was not returning the correct results (it returned all Rvvup Refunds)
        // As a result, we used `meta_key` & `meta_value` keys
        $args = [
            "type" => RvvupRefund::TYPE,
            "parent" => $orderId,
            "meta_key" => "_woocommerce_refund_id",
            "meta_value" => $refundId,
            "limit" => 1, // We expect a single result anyway because Rvvup Refund ID is unique
        ];

        $rvvupRefunds = wc_get_orders($args);

        // Placeholder
        $rvvupRefund = null;

        if (!empty($rvvupRefunds)) {
            /** @var \Rvvup\Payments\Model\PostTypes\RvvupRefund $rvvupRefund */
            $rvvupRefund = reset($rvvupRefunds);
        }

        wp_cache_set($cacheKey, $rvvupRefund, $this->cacheGroup);

        return $rvvupRefund;
    }

    /**
     * Get an order's Rvvup Refunds
     *
     * @param int $orderId
     * @return \Rvvup\Payments\Model\PostTypes\RvvupRefund[]
     */
    public function getPendingByOrderId(int $orderId): array
    {
        $cacheKey = WC_Cache_Helper::get_cache_prefix($this->cacheGroup) . "rvvup-refund_order-id-" . $orderId;

        $cachedData = wp_cache_get($cacheKey, $this->cacheGroup);

        if ($cachedData !== false) {
            return $cachedData;
        }

        $args = [
            "post_type" => RvvupRefund::TYPE,
            "parent" => $orderId,
            "status" => "wc-pending",
            "limit" => -1,
        ];

        $rvvupRefunds = wc_get_orders($args);

        wp_cache_set($cacheKey, $rvvupRefunds, $this->cacheGroup);

        return $rvvupRefunds;
    }

    /**
     * Get the total of all pending RvvupRefund entities by their parent order ID.
     *
     * @param int $orderId
     * @return float
     */
    public function getPendingTotalByOrderId(int $orderId): float
    {
        $pendingRefunds = $this->getPendingByOrderId($orderId);

        $total = 0;

        foreach ($pendingRefunds as $pendingRefund) {
            $total += (float) $pendingRefund->get_amount();
        }

        return $total;
    }

    /**
     * Get a RvvupRefund entity by Rvvup's ID.
     *
     * @param string $refundId
     * @return \Rvvup\Payments\Model\PostTypes\RvvupRefund|null
     */
    public function getByRefundId(string $refundId): ?RvvupRefund
    {
        $cacheKey = WC_Cache_Helper::get_cache_prefix($this->cacheGroup) . "rvvup-refund_rvvup-refund-id-" . $refundId;

        $cachedData = wp_cache_get($cacheKey, $this->cacheGroup);

        if ($cachedData !== false) {
            return $cachedData;
        }

        // Using meta_query array arguments was not returning the correct results (it returned all Rvvup Refunds)
        // As a result, we used `meta_key` & `meta_value` keys
        $args = [
            "type" => RvvupRefund::TYPE,
            "meta_key" => "_refund_id",
            "meta_value" => $refundId,
            "limit" => 1, // We expect a single result anyway, as Rvvup's Refund ID is unique.
        ];

        $rvvupRefunds = wc_get_orders($args);

        // Placeholder
        $rvvupRefund = null;

        if (!empty($rvvupRefunds)) {
            $rvvupRefund = reset($rvvupRefunds);
        }

        wp_cache_set($cacheKey, $rvvupRefund, $this->cacheGroup);

        return $rvvupRefund;
    }
}
