<?php declare(strict_types=1);

namespace Rvvup\Payments\Service;

use Rvvup\Payments\Exceptions\ApiException;
use Rvvup\Payments\Exceptions\ConfigException;
use Rvvup\Payments\Lib\Versions;
use Rvvup\Payments\Sdk\Curl;
use Rvvup\Payments\Sdk\Factories\Inputs\RefundCreateInputFactory;
use Rvvup\Payments\Sdk\GraphQlSdk;
use Rvvup\Payments\Sdk\Rest\RvvupClient;
use Rvvup\Payments\Sdk\Rest\RvvupClientOptions;
use WC_Order;

/**
 * Previously we had a SdkInstanceManager class which handles instantiation of the SDK, however we realised we wanted
 * to perform some additional actions whilst performing the API calls that we wanted to perform some additional logic
 * so decided on a Proxy class that provided some additional methods around the SDK (we are mainly caching responses).
 */
class SdkProxy
{
    public const BACKEND_API_URL = "backend_api_url";
    public const MERCHANT_ID = "merchant_id";
    public const AUTH_TOKEN = "auth_token";

    /** @var GraphQlSdk */
    private static $sdk;

    /** @var RvvupClient */
    private static $rvvupClient;

    /** @var Cache */
    private static $cache;

    /** @var array */
    private static $responses = [];

    const PRICE_CACHE_KEY = "rvvup_price_thresholds_cache";

    /**
     * @return \Rvvup\Payments\Sdk\GraphQlSdk
     * @throws \Rvvup\Payments\Exceptions\ConfigException
     */
    private static function get(): GraphQlSdk
    {
        if (self::$sdk !== null) {
            return self::$sdk;
        }

        $config = self::getConfig();

        self::$sdk = new GraphQlSdk(
            $config[self::BACKEND_API_URL],
            $config[self::MERCHANT_ID],
            $config[self::AUTH_TOKEN],
            Versions::getUserAgent(),
            new Curl(),
            wc_get_logger(),
            (bool) $config["debug"]
        );

        return self::$sdk;
    }

    private static function getRvvupClient(): RvvupClient
    {
        if (self::$rvvupClient !== null) {
            return self::$rvvupClient;
        }

        $config = self::getConfig();

        self::$rvvupClient = new RvvupClient(
            $config["jwt"],
            new RvvupClientOptions(null, null, Versions::getUserAgent())
        );

        return self::$rvvupClient;
    }
    /**
     * @return Cache
     */
    private static function getCacheService(): Cache
    {
        if (self::$cache !== null) {
            return self::$cache;
        }

        self::$cache = new Cache();

        return self::$cache;
    }

    /** Get plugin config
     * @throws ConfigException
     */
    private static function getConfig(): array
    {
        $config = get_option("woocommerce_rvvup_gateway_settings", null);

        if (!is_array($config)) {
            throw new ConfigException("No Rvvup gateway config available");
        }

        if (
            empty($config[self::BACKEND_API_URL]) ||
            empty($config[self::MERCHANT_ID]) ||
            empty($config[self::AUTH_TOKEN])
        ) {
            throw new ConfigException("Rvvup gateway config is missing, please fill all the data");
        }

        return $config;
    }

    /**
     * When loading the acceptable threshold for the payment methods we ideally want to read from a locally cached
     * copy, but we also want a current value. As a compromise we read from our cached copy 1st, if the ttl has expired
     * we read from the API with a short timeout. If the HTTP client returns due to exceeding the timeout we will use
     * the cached copy.
     * @return array
     * @throws \Exception
     */
    public static function getThresholds()
    {
        $cachedThreshold = get_transient(self::PRICE_CACHE_KEY);
        $args = ["0", get_woocommerce_currency(), ["timeout" => 1]];
        if (!is_array($cachedThreshold) || count($cachedThreshold) <= 0) {
            //cache is not primed
            $cachedThreshold = self::get()->getMethods(...$args);
            $processed = [];
            foreach ($cachedThreshold as $processor) {
                if (is_array($processor["limits"])) {
                    $processed[$processor["name"]] = $processor["limits"]["total"];
                    if (!isset($processed["ttl"]) && isset($processor["limits"]["expiresAt"])) {
                        $processed["ttl"] = $processor["limits"]["expiresAt"];
                    }
                }
            }
            set_transient(self::PRICE_CACHE_KEY, $processed);
            unset($cachedThreshold["ttl"]);
            return $processed;
        } else {
            $now = new \DateTime();
            if (isset($cachedThreshold["ttl"])) {
                $cachedTtl = new \DateTime($cachedThreshold["ttl"]);
                if ($cachedTtl > $now) {
                    unset($cachedThreshold["ttl"]);
                    return $cachedThreshold;
                }
            }
            $response = self::get()->getMethods(...$args);
            // We store the transient with no ttl, so we have a fallback value
            if (count($response) === 0) {
                // Likely due to timeout, DNS error or no connectivity, we should use the cached value
                unset($cachedThreshold["ttl"]);
                return $cachedThreshold;
            }
            $processed = [];
            foreach ($response as $processor) {
                if (is_array($processor["limits"])) {
                    $processed[$processor["name"]] = $processor["limits"]["total"];
                    if (!isset($processed["ttl"]) && isset($processor["limits"]["expiresAt"])) {
                        $processed["ttl"] = $processor["limits"]["expiresAt"];
                    }
                }
            }
            set_transient(self::PRICE_CACHE_KEY, $processed);
            unset($processed["ttl"]);
            return $processed;
        }
    }

    /**
     * When querying the available methods we use the result to update our cached limits/ttl
     * @param string $cartTotal
     * @return array
     * @throws \Exception
     */
    public static function getMethods(string $cartTotal)
    {
        if (!isset(self::$responses[$cartTotal])) {
            self::$responses[$cartTotal] = self::get()->getMethods($cartTotal, get_woocommerce_currency());
        }
        return self::$responses[$cartTotal];
    }

    /**
     * @param $orderData
     * @return mixed
     * @throws \Exception
     */
    public static function createOrder($orderData)
    {
        return self::get()->createOrder($orderData);
    }

    /**
     * {@inheritdoc}
     */
    public static function createPayment($paymentData)
    {
        return self::get()->createPayment($paymentData);
    }

    /**
     * {@inheritdoc}
     */
    public static function updateOrder($data)
    {
        return self::get()->updateOrder($data);
    }

    /**
     * @param $orderId
     * @return false|mixed
     * @throws \Exception
     */
    public static function getOrder($orderId)
    {
        return self::get()->getOrder($orderId);
    }

    /**
     * @param WC_Order $order
     * @return false|mixed
     * @throws ConfigException
     */
    public static function isOrderRefundable(WC_Order $order)
    {
        $orderId = $order->get_transaction_id();
        $status = $order->get_status();
        $cacheService = self::getCacheService();
        $result = $cacheService->get($orderId, "refund", $status);

        if ($result === false) {
            $result = json_encode(["available" => self::get()->isOrderRefundable($orderId)]);
            $cacheService->set($orderId, "refund", $result, $status);
        }

        return json_decode($result, true)["available"];
    }

    /**
     * @param $orderId
     * @param $amount
     * @param $reason
     * @param $idempotency
     * @return false|mixed
     * @throws \Exception
     */
    public static function refundOrder($orderId, $amount, $reason, $idempotency)
    {
        $value = self::get()->refundOrder($orderId, $amount, $reason, $idempotency);
        $order = wc_get_order($orderId);
        self::getCacheService()->clear($orderId, $order->get_status());
        return $value;
    }

    /**
     * {@inheritdoc}
     */
    public static function isOrderVoidable(WC_Order $order)
    {
        $orderId = $order->get_transaction_id();
        $status = $order->get_status();
        $cacheService = self::getCacheService();

        $value = $cacheService->get($orderId, "void", $status);

        if ($value === false) {
            $value = json_encode(["available" => self::get()->isOrderVoidable($orderId)]);
            $cacheService->set($orderId, "void", $value, $status);
        }

        return json_decode($value, true)["available"];
    }

    /**
     * @param WC_Order $order
     * @param string $paymentId
     * @return false|mixed
     * @throws ConfigException
     */
    public static function voidPayment(WC_Order $order, string $paymentId)
    {
        $id = (string) $order->get_transaction_id();
        $result = self::get()->voidPayment($id, $paymentId);
        self::getCacheService()->clear((string) $order->get_id(), $order->get_status());
        return $result;
    }

    /**
     * @return bool
     * @throws \Exception
     */
    public static function ping(): bool
    {
        return self::get()->ping();
    }

    /**
     * @param string $url
     * @return bool
     */
    public static function registerWebhook(string $url): bool
    {
        return self::get()->registerWebhook($url);
    }

    /**
     * @param string $eventType
     * @param string $reason
     * @return void
     * @throws \Exception
     */
    public static function createEvent(string $eventType, string $reason): void
    {
        $environmentVersions = Versions::getEnvironmentVersions();

        $data = [
            "plugin" => $environmentVersions["rvvp_plugin_version"],
            "php" => $environmentVersions["php_version"],
            "wordpress" => $environmentVersions["wordpress_version"],
            "woocommerce" => $environmentVersions["woocommerce_version"],
        ];

        self::get()->createEvent($eventType, $reason, $data);
    }

    /**
     * @param string $orderId
     * @param string $amount
     * @param string $currency
     * @param string $idempotency
     * @param string|null $reason
     * @return false|array
     * @throws \Exception
     * @throws \JsonException
     * @throws \Rvvup\Payments\Exceptions\ApiException
     * @throws \Rvvup\Payments\Sdk\Exceptions\NetworkException
     */
    public static function refundCreate(
        string $orderId,
        string $amount,
        string $currency,
        string $idempotency,
        ?string $reason = null
    ) {
        $factory = new RefundCreateInputFactory();

        $result = static::get()->refundCreate($factory->create($orderId, $amount, $currency, $idempotency, $reason));

        // If false, throw custom exception.
        if ($result === false) {
            throw new ApiException("Failed to create refund");
        }

        return $result;
    }

    /**
     * @param string $orderId
     * @return array
     * @throws \Exception
     * @throws \JsonException
     * @throws \Rvvup\Payments\Exceptions\ApiException
     * @throws \Rvvup\Payments\Exceptions\ConfigException
     * @throws \Rvvup\Payments\Sdk\Exceptions\NetworkException
     */
    public static function getOrderRefunds(string $orderId): array
    {
        $result = static::get()->getOrderRefunds($orderId);

        // If false, throw custom exception.
        if ($result === false) {
            throw new ApiException("Failed to get order refunds");
        }

        return $result;
    }

    /**
     * Cancel rvvup payment
     * @param string $paymentId
     * @param string $orderId
     * @return array
     * @throws ConfigException
     */
    public static function cancelPayment(string $paymentId, string $orderId): array
    {
        return static::get()->cancelPayment($paymentId, $orderId);
    }

    /**
     * Confirm rvvup card authorization
     * @param string $paymentId
     * @param string $orderId
     * @param string $authorizationResponse
     * @param string|null $threeDSecureResponse
     * @return array
     * @throws ConfigException
     */
    public static function confirmCardAuthorization(
        string $paymentId,
        string $orderId,
        string $authorizationResponse,
        ?string $threeDSecureResponse
    ): array {
        return static::get()->confirmCardAuthorization(
            $paymentId,
            $orderId,
            $authorizationResponse,
            $threeDSecureResponse
        );
    }

    public static function createShipmentTracking(string $paymentSessionId, array $trackingData): array
    {
        return static::getRvvupClient()
            ->shipmentTrackings()
            ->create($paymentSessionId, $trackingData);
    }
}
