<?php

namespace Rvvup\Payments\Service;

use Exception;
use Rvvup\Payments\Facade\WCTaxFacade;

class TaxRateCalculator
{
    /** @var WCTaxFacade */
    private $taxFacade;

    /** @var array<string, string> */
    private $taxRateCache = [];

    /**
     * @var TaxRateCalculator|null
     */
    private static $instance = null;

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct(WCTaxFacade $taxFacade)
    {
        $this->taxFacade = $taxFacade;
    }

    /**
     * Get the singleton instance
     *
     * @param null $taxFacade
     * @return TaxRateCalculator
     */
    public static function getInstance($taxFacade = null): TaxRateCalculator
    {
        if (self::$instance === null || $taxFacade !== null) {
            self::$instance = new self($taxFacade ?? WCTaxFacade::getInstance());
        }

        return self::$instance;
    }

    public function getItemTaxRate($order, $product): ?string
    {
        try {
            if (!$this->taxFacade->isTaxEnabled()) {
                return null;
            }
            if ($order === null) {
                return null;
            }
            if ($product === null) {
                return null;
            }
            // Get the product tax class
            $taxClass = $product->get_tax_class();

            // Use current customer location if no address provided
            $address = $this->getCurrentCustomerAddress($order);

            // Create cache key based on tax class and location
            $cacheKey = $this->getCacheKey($taxClass, $address);

            if (isset($this->taxRateCache[$cacheKey])) {
                return $this->taxRateCache[$cacheKey];
            }

            // Get tax rates for the product's tax class and location
            $request = $address;
            $request["tax_class"] = $taxClass;
            $taxRates = $this->taxFacade->findRates($request);
            if (empty($taxRates)) {
                $this->taxRateCache[$cacheKey] = null;
                return null;
            }

            // Calculate total tax rate
            $totalRate = 0;
            foreach ($taxRates as $taxRate) {
                $totalRate += (float) $taxRate["rate"];
            }

            $rateString = (string) $totalRate;
            $this->taxRateCache[$cacheKey] = $rateString;
            return $rateString;
        } catch (Exception $e) {
            return null;
        }
    }

    private function getCurrentCustomerAddress($order): array
    {
        $address = [
            "country" => $order->get_shipping_country() ?: $order->get_billing_country() ?: "",
            "state" => $order->get_shipping_state() ?: $order->get_billing_state() ?: "",
            "postcode" => $order->get_shipping_postcode() ?: $order->get_billing_postcode() ?: "",
            "city" => $order->get_shipping_city() ?: $order->get_billing_city() ?: "",
        ];

        if (!empty($address["country"])) {
            return $address;
        }

        $customer = WC()->customer;

        if ($customer) {
            // Try to get shipping address first, fallback to billing
            $address = [
                "country" => $customer->get_shipping_country() ?: $customer->get_billing_country() ?: "",
                "state" => $customer->get_shipping_state() ?: $customer->get_billing_state() ?: "",
                "postcode" => $customer->get_shipping_postcode() ?: $customer->get_billing_postcode() ?: "",
                "city" => $customer->get_shipping_city() ?: $customer->get_billing_city() ?: "",
            ];
            if (!empty($address["country"])) {
                return $address;
            }
        }

        return [
            "country" => WC()->countries->get_base_country() ?: "",
            "state" => WC()->countries->get_base_state() ?: "",
            "postcode" => WC()->countries->get_base_postcode() ?: "",
            "city" => WC()->countries->get_base_city() ?: "",
        ];
    }

    /**
     * Generate a cache key for tax rate lookup
     *
     * @param string $taxClass
     * @param array $address
     * @return string
     */
    private function getCacheKey(string $taxClass, array $address): string
    {
        return md5(
            serialize([
                "taxClass" => $taxClass,
                "country" => $address["country"] ?? "",
                "state" => $address["state"] ?? "",
                "postcode" => $address["postcode"] ?? "",
                "city" => $address["city"] ?? "",
            ])
        );
    }
}
