<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Exception;
use Rvvup\Payments\Gateway\Dynamic;
use Rvvup\Payments\Traits\GatewayTrait;
use WC_Product;
use WC_Product_Variable;
use WC_Product_Variation;

class ViewManager
{
    use GatewayTrait;

    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * @return void
     */
    public function init(): void
    {
        add_action("woocommerce_after_add_to_cart_form", [$this, "renderProductPageRvvupExpressPaymentContainer"]);
        add_action("woocommerce_review_order_before_submit", [$this, "renderCheckoutPaymentMethodsMessaging"]);
        add_action("woocommerce_review_order_after_submit", [
            $this,
            "renderCheckoutRvvupExpressPaymentCancellationAction",
        ]);
        add_filter("woocommerce_short_description", [$this, "renderProductPagePaymentMethodsMessaging"]);
        add_filter("woocommerce_get_price_html", [$this, "renderVariableProductPagePaymentMethodsMessaging"], 10, 2);
        add_filter(
            "woocommerce_available_variation",
            [$this, "renderEmptyPriceVariationProductPagePaymentMethodsMessaging"],
            10,
            3
        );
        add_action("woocommerce_before_cart_totals", [ClearpayMessaging::class, "addClearpayLogoCart"]);
    }

    /**
     * Render express payment container.
     *
     * @return void
     */
    public function renderProductPageRvvupExpressPaymentContainer(): void
    {
        if (!RvvupAvailable::getInstance()->isEnabled()) {
            return;
        }

        global $product;

        if (!($product instanceof WC_Product) || !is_product()) {
            return;
        }

        if (!RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductType($product->get_type(), false)) {
            return;
        }

        $position = GatewaySettingsManager::getInstance()->getMobileModalPosition() ?? "bottom";

        echo "<rvvup-express-payments></rvvup-express-payments>";
        echo '<div id="rvvup-modal-express" class="rvvup-modal">
    <div class="rvvup-dialog slide-in-' .
            $position .
            '" role="dialog" aria-modal="true">
        <iframe class="rvvup-iframe" src=""></iframe>
    </div>
</div>';
    }

    /**
     * Display a message & url that allows you to cancel express payment during checkout (if we're on an express order).
     * Message should be displayed below the `Place Order` Button.
     *
     * @return void
     */
    public function renderCheckoutRvvupExpressPaymentCancellationAction(): void
    {
        $rvvupExpressOrder = SessionManager::getInstance()->getRvvupExpressOrder();

        if (!is_array($rvvupExpressOrder) || !isset($rvvupExpressOrder["id"])) {
            return;
        }

        echo '<p id="rvvup-express-payments-cancellation" class="has-text-align-center">';
        printf(
            esc_html__(
                'You are currently paying with %3$s. If you want to cancel
                            this process, please click %1$shere%2$s.',
                "rvvup-for-woocommerce"
            ),
            '<a id="rvvup-express-payments-cancellation-link" href="">',
            "</a>",
            esc_html(ucfirst($rvvupExpressOrder["payment_method_code"]))
        );
        echo "</p>";
    }

    /**
     * Render the payment methods custom messaging in the Checkout page
     *
     * @return void
     */
    public function renderCheckoutPaymentMethodsMessaging(): void
    {
        if (!is_checkout() || WC()->cart === null || !RvvupAvailable::getInstance()->isEnabled()) {
            return;
        }

        foreach (
            WC()
                ->payment_gateways()
                ->get_available_payment_gateways()
            as $paymentGateway
        ) {
            if (!$this->isRvvupDynamicGateway($paymentGateway->id)) {
                continue;
            }

            $rvvupMethodName = mb_strtolower($this->getRvvupMethodName($paymentGateway->id));

            if ($rvvupMethodName !== "paypal") {
                continue;
            }

            echo $this->getPayPalMessaging($paymentGateway, (string) WC()->cart->get_total(null), "checkout");
        }
    }

    /**
     * Render the payment methods custom messaging in the Product Page.
     *
     * @param $html
     * @return string
     */
    public function renderProductPagePaymentMethodsMessaging($html): string
    {
        global $product;

        if (!($product instanceof WC_Product) || !is_product()) {
            return $html;
        }

        return $this->getProductPagePaymentMethodsMessaging($html, $product, false);
    }

    /**
     * Render the payment methods custom messaging in the Product Page.
     *
     * @param $prices
     * @param \WC_Product $product
     * @return string
     */
    public function renderVariableProductPagePaymentMethodsMessaging($prices, WC_Product $product): string
    {
        $prices = is_null($prices) ? "" : $prices;

        // If not a product variation, no action.
        if (!$product->is_type("variation")) {
            return $prices;
        }

        return $this->getProductPagePaymentMethodsMessaging($prices, $product);
    }

    /**
     * @param array $data
     * @param \WC_Product_Variable $variable
     * @param \WC_Product_Variation $variation
     * @return array
     */
    public function renderEmptyPriceVariationProductPagePaymentMethodsMessaging(
        array $data,
        WC_Product_Variable $variable,
        WC_Product_Variation $variation
    ): array {
        // If the Price html is already customised, the element is not empty,
        // `renderVariableProductPagePaymentMethodsMessaging` should have already been triggered by the relevant filter.
        if (!isset($data["price_html"]) || !empty($data["price_html"])) {
            return $data;
        }

        $data["price_html"] = $this->getProductPagePaymentMethodsMessaging($data["price_html"], $variation);

        return $data;
    }

    /**
     * Get the payment methods messaging, filter by area loaded (main or variant prices)
     *
     * @param string $html
     * @param \WC_Product $product
     * @param bool $loadVariant
     * @return string
     */
    private function getProductPagePaymentMethodsMessaging(
        string $html,
        WC_Product $product,
        bool $loadVariant = true
    ): string {
        if (!RvvupAvailable::getInstance()->isEnabled()) {
            return $html;
        }

        // If method is called for the main product page rendered, do not display messaging if Variable product.
        if (!$loadVariant && $product->is_type("variable")) {
            return $html;
        }

        if (
            !RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductType($product->get_type(), $loadVariant)
        ) {
            return $html;
        }

        $paymentMethodsMessagingHtml = [];

        foreach (
            WC()
                ->payment_gateways()
                ->payment_gateways()
            as $paymentGateway
        ) {
            if (!$this->isRvvupDynamicGateway($paymentGateway->id)) {
                continue;
            }

            if (
                !RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductId(
                    (string) $product->get_id(),
                    $paymentGateway->id
                )
            ) {
                continue;
            }
            /** @var \Rvvup\Payments\Gateway\Dynamic $paymentGateway */
            switch (mb_strtolower($this->getRvvupMethodName($paymentGateway->id))) {
                case "clearpay":
                    try {
                        $paymentMethodsMessagingHtml[] = !$loadVariant
                            ? ClearpayMessaging::addClearpayLogoPdp("")
                            : ClearpayMessaging::filterVariantPrices("", $product);
                    } catch (Exception $e) {
                        $paymentMethodsMessagingHtml[] = "";
                    }
                    break;
                case "paypal":
                    $paymentMethodsMessagingHtml[] = $this->getPayPalMessaging(
                        $paymentGateway,
                        (string) $product->get_price(),
                        "product"
                    );
                    break;
                case "zopa_retail_finance":
                    $paymentMethodsMessagingHtml[] = $this->getZopaRetailFinanceMessaging(
                        $paymentGateway,
                        (string) $product->get_price(),
                        "product"
                    );
                    break;
                default:
                    break;
            }
        }

        $html .= implode("", $paymentMethodsMessagingHtml);

        return $html;
    }

    /**
     * Get PayPal's Pay Later Messaging.
     *
     * @param Dynamic $paymentMethod
     * @param string $value
     * @param string|null $setting
     * @return string
     */
    private function getPayPalMessaging(Dynamic $paymentMethod, string $value, ?string $setting = null): string
    {
        $availableSettings = ["product", "checkout"];

        if (!in_array($setting, $availableSettings, true)) {
            return "";
        }

        $gatewaySettings = GatewaySettingsManager::getInstance();

        if ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/enabled") !== true) {
            return "";
        }

        $placeholderHtml = "<div data-pp-message ";

        // Hide messaging on original checkout load.
        if ($setting === "checkout") {
            $placeholderHtml .= 'id="checkout-messaging-' . $paymentMethod->id . '" ';
            $placeholderHtml .= 'class="checkout-messaging-rvvup_gateway" ';
            $placeholderHtml .= 'style="display:none;" ';
        }

        $placeholderHtml .= 'data-pp-amount="' . $value . '" ';
        $placeholderHtml .=
            'data-pp-style-layout="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/layout/value") ?? "text") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-logo-type="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/logoType/value") ??
                "primary") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-logo-position="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/logoPosition/value") ??
                "left") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-text-color="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/textColor/value") ??
                "white") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-text-size="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/textSize") ?? "12") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-text-align="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/textAlignment/value") ??
                "left") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-color="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/color/value") ?? "blue") .
            '" ';
        $placeholderHtml .=
            'data-pp-style-ratio="' .
            ($gatewaySettings->getApiConfig($paymentMethod, $setting . "/payLaterMessaging/ratio/value") ?? "1x1") .
            '"';
        $placeholderHtml .= ">";
        $placeholderHtml .= "</div>";

        return $placeholderHtml;
    }

    private function getZopaRetailFinanceMessaging(
        Dynamic $paymentMethod,
        string $value,
        ?string $setting = null
    ): string {
        $availableSettings = ["product"];

        if (!in_array($setting, $availableSettings, true)) {
            return "";
        }

        global $product;
        if (!($product instanceof WC_Product) || !is_product()) {
            return "";
        }

        // If item product type is not enabled for Rvvup, no message.
        if (!RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductType($product->get_type(), false)) {
            return "";
        }

        // If item product is restricted, no message.
        if (
            !RvvupAvailable::getInstance()->isRvvupAvailableForProductByProductId(
                (string) $product->get_id(),
                "ZOPA_RETAIL_FINANCE"
            )
        ) {
            return "";
        }

        return '<div id="rvvup-zrf-widget-container"></div>
        <script>
        function initZrfWidget() {
            let showHideWidget = async function (el, rvvupWidgetInstance) {
                if (!(await rvvupWidgetInstance.canDisplayWidget())) {
                    el.style.display = "none";
                    return;
                }
        
                el.style.display = "block";
            }
            if (window.rvvup_sdk) {
                window.rvvup_sdk.createWidget("ZOPA_RETAIL_FINANCE", {
                    settings: { 
                        context: "product",
                    },
                    total: {
                        currency: "' .
            get_woocommerce_currency() .
            '",
                        amount: "' .
            $value .
            '"
                    },
                }).then((widget) => {
                    widget.on("ready", async () => {
                        const el = document.getElementById("rvvup-zrf-widget-container");
                        if (!el) return;
                        await widget.mount({selector: el});
                        await showHideWidget(el, widget);
                    });
                }).catch((error) => {
                    console.error("Error creating Rvvup widget:", error);
                });
            }
        }
        
        initZrfWidget();
        </script>';
    }
}
