<?php

declare(strict_types=1);

namespace Rvvup\Payments\Service;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Exception;
use Rvvup\Payments\Exceptions\ApiException;
use Rvvup\Payments\Gateway\Rvvup;
use Rvvup\Payments\Model\PostTypes\RvvupRefund;
use WC_Order;

class WebhookManager
{
    /**
     * @var self
     */
    private static $instance;

    /**
     * For this class we use the singleton pattern to avoid redundant DB I/O
     *
     * @return static
     */
    public static function getInstance(): self
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Process the Refund Completed event webhook.
     *
     * @param string $orderId
     * @param string $refundId
     * @return bool
     */
    public function refundCompleted(string $orderId, string $refundId): bool
    {
        try {
            $rvvupRefund = RvvupRefundRepository::getInstance()->getByRefundId($refundId);

            // No action if we cannot find a matching result or Refund is not pending.
            if ($rvvupRefund === null || !$rvvupRefund->has_status("pending")) {
                return true;
            }

            // Get the WC Order & return if none (should not happen, only if deleted).
            $order = wc_get_order($rvvupRefund->get_parent_id());

            if (!$order) {
                return true;
            }

            // Get the Order's payments & refunds.
            $result = SdkProxy::getOrderRefunds($orderId);

            // Get the matching refund from the result set & return if none found.
            $apiRefund = $this->findRefundInPaymentRefunds($result["payments"], $refundId);

            // No action if non found or status is not set (should not happen).
            if ($apiRefund === null || !isset($apiRefund["status"])) {
                return true;
            }

            switch ($apiRefund["status"]) {
                case Rvvup::REFUND_STATUS_SUCCEEDED:
                    return $this->processSuccessfulRefundResult($order, $rvvupRefund);
                case Rvvup::REFUND_STATUS_FAILED:
                    return $this->processFailedRefundResult($order, $rvvupRefund);
                case Rvvup::REFUND_STATUS_PENDING:
                default:
                    return true;
            }
        } catch (ApiException $ex) {
            if (!GatewaySettingsManager::getInstance()->isDebugEnabled()) {
                return true;
            }

            LoggerManager::getInstance()->debug(
                "API error on Refund Complete webhook with message: " . $ex->getMessage(),
                [
                    "order_id" => $orderId,
                    "refund_id" => $refundId,
                ]
            );

            return true;
        } catch (Exception $ex) {
            LoggerManager::getInstance()->error(
                "Exception thrown on Refund Complete webhook with message: " . $ex->getMessage(),
                [
                    "order_id" => $orderId,
                    "refund_id" => $refundId,
                ]
            );

            return false;
        }
    }

    /**
     * Loop through API result payments & get the matching refund.
     *
     * @param array $payments
     * @param string $refundId
     * @return array|null
     */
    private function findRefundInPaymentRefunds(array $payments, string $refundId): ?array
    {
        foreach ($payments as $payment) {
            if (!isset($payment["refunds"])) {
                continue;
            }

            foreach ($payment["refunds"] as $refund) {
                if (!isset($refund["id"]) || $refund["id"] !== $refundId) {
                    continue;
                }

                return $refund;
            }
        }

        return null;
    }

    /**
     * Update the Rvvup Refund status, the order status, add notes etc. on successful refund result.
     *
     * @param \WC_Order $order
     * @param \Rvvup\Payments\Model\PostTypes\RvvupRefund $rvvupRefund
     * @return bool
     */
    private function processSuccessfulRefundResult(WC_Order $order, RvvupRefund $rvvupRefund): bool
    {
        $rvvupRefund->set_status("wc-completed");
        $rvvupRefund->save();

        // Add order history note.
        $note = sprintf(
            __('Refund for %1$s succeeded - Refund ID: %2$s', "rvvup-for-woocommerce"),
            wc_price($rvvupRefund->get_amount(), ["currency" => $rvvupRefund->get_currency()]),
            $rvvupRefund->get_refund_id()
        );

        $order->add_order_note($note);

        RefundManager::getInstance()->triggerRefundNotifications(
            $order->get_id(),
            $rvvupRefund->get_woocommerce_refund_id(),
            (bool) $rvvupRefund->get_is_full_refund()
        );

        // If this was not a full refund or order status is not pending refund, no further action.
        if (
            (bool) $rvvupRefund->get_is_full_refund() === false ||
            !$order->has_status(ltrim(OrderStatusManager::STATUS_REFUND_PENDING, "wc-"))
        ) {
            return true;
        }

        $newStatus = apply_filters(
            "woocommerce_order_fully_refunded_status",
            "refunded",
            $order->get_id(),
            $rvvupRefund->get_woocommerce_refund_id()
        );

        if (!$newStatus) {
            return true;
        }

        $order->update_status($newStatus);

        return true;
    }

    /**
     * Update the Rvvup Refund status, the order status, add notes etc. on failed refund result.
     *
     * @param \WC_Order $order
     * @param \Rvvup\Payments\Model\PostTypes\RvvupRefund $rvvupRefund
     * @return bool
     */
    private function processFailedRefundResult(WC_Order $order, RvvupRefund $rvvupRefund): bool
    {
        $rvvupRefund->set_status("wc-failed");
        $rvvupRefund->save();

        // If not WC Refund, return.
        if (get_post_type($rvvupRefund->get_woocommerce_refund_id()) !== "shop_order_refund") {
            return true;
        }

        // Delete WC Refund on Refund Failed webhook.
        $wcRefund = wc_get_order($rvvupRefund->get_woocommerce_refund_id());
        $wcRefund->delete(true);

        // Trigger hooks.
        do_action(
            "woocommerce_refund_deleted",
            $rvvupRefund->get_woocommerce_refund_id(),
            $rvvupRefund->get_parent_id()
        );

        // Add order history note.
        $note = sprintf(
            __('Refund for %1$s failed & deleted - Refund ID: %2$s', "rvvup-for-woocommerce"),
            wc_price($rvvupRefund->get_amount(), ["currency" => $rvvupRefund->get_currency()]),
            $rvvupRefund->get_refund_id()
        );

        $order->add_order_note($note);

        // If this was not a full refund or order status is not pending refund, no further action.
        if (
            (bool) $rvvupRefund->get_is_full_refund() === false ||
            !$order->has_status(ltrim(OrderStatusManager::STATUS_REFUND_PENDING, "wc-"))
        ) {
            return true;
        }

        $order->update_status("wc-refund-failed");

        return true;
    }
}
