<?php

declare(strict_types=1);

namespace Rvvup\Payments\Traits;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

use Rvvup\Payments\Gateway\Fallback;

trait GatewayTrait
{
    /**
     * Check whether the payment gateway is a Rvvup gateway.
     *
     * @param string $method
     * @return bool
     */
    protected function isRvvupGateway(string $method): bool
    {
        return strpos($method, "rvvup_gateway_") === 0;
    }

    /**
     * Check whether the payment method is the fallback Rvvup gateway.
     *
     * @param string $method
     * @return bool
     */
    protected function isRvvupFallbackGateway(string $method): bool
    {
        return $method === Fallback::ID;
    }

    /**
     * Rvvup Gateways loaded as Dynamic (API generated) have a specific prefix.
     * Exclude the fallback gateway.
     *
     * @param string $method
     * @return bool
     */
    protected function isRvvupDynamicGateway(string $method): bool
    {
        return $this->isRvvupGateway($method) && !$this->isRvvupFallbackGateway($method);
    }

    /**
     * Get the gateway's method name as named in Rvvup (removing our custom prefix).
     *
     * @param string $method
     * @return string
     */
    protected function getRvvupMethodName(string $method): string
    {
        return str_replace("rvvup_gateway_", "", $method);
    }
}
