<?php
/**
 * Plugin Name: Rvvup for WooCommerce
 * Description: Accept payments with Rvvup.
 * Version: 1.1.0
 * Author: Rvvup
 * Author URI: https://www.rvvup.com
 * Text Domain: rvvup-for-woocommerce
 * Tested up to: 6.6.2
 * WC tested up to: 9.3.1
 */

use Automattic\WooCommerce\Utilities\FeaturesUtil;
use Rvvup\Payments\Actions\ExpressCheckoutCartModification;
use Rvvup\Payments\Actions\ExpressCheckoutLoad;
use Rvvup\Payments\Controller\CardSubmitCallbackEndpoint;
use Rvvup\Payments\Controller\ExpressDeleteEndpoint;
use Rvvup\Payments\Controller\ExpressEndpoint;
use Rvvup\Payments\Controller\ExpressUpdateEndpoint;
use Rvvup\Payments\Controller\LogController;
use Rvvup\Payments\Controller\PayOrderEndpoint;
use Rvvup\Payments\Controller\ReturnEndpoint;
use Rvvup\Payments\Controller\VoidEndpoint;
use Rvvup\Payments\Controller\WebhookEndpoint;
use Rvvup\Payments\Gateway\Rvvup;
use Rvvup\Payments\Gateway\Rvvup_WC_Blocks_Support;
use Rvvup\Payments\Lib\Versions;
use Rvvup\Payments\Model\AssetLoader;
use Rvvup\Payments\Model\ClosedBetaUpdater;
use Rvvup\Payments\Service\AdminViewManager;
use Rvvup\Payments\Service\CheckoutPopulate;
use Rvvup\Payments\Service\EventManager;
use Rvvup\Payments\Service\GatewayLoader;
use Rvvup\Payments\Service\OrderStatusManager;
use Rvvup\Payments\Service\PostTypesManager;
use Rvvup\Payments\Service\RefundManager;
use Rvvup\Payments\Service\RvvupRefundManager;
use Rvvup\Payments\Service\RvvupRefundRepository;
use Rvvup\Payments\Service\ViewManager;
use Rvvup\Payments\Service\VoidOrder;

if (!defined("ABSPATH")) {
    exit(); // Exit if accessed directly
}

define("RVVUP_PLUGIN_PATH", plugin_dir_path(__FILE__));
define("RVVUP_PLUGIN_FILE", __FILE__);
define("RVVUP_PLUGIN_URL", plugin_dir_url(__FILE__));

// We want that to be loaded first, even if we don't have woocommerce or rvvup enabled
add_action("before_woocommerce_init", function () {
    if (class_exists("\Automattic\WooCommerce\Utilities\FeaturesUtil")) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility("custom_order_tables", __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility("cart_checkout_blocks", __FILE__, true);
    }
});

// Check if WooCommerce is active

if (in_array("woocommerce/woocommerce.php", apply_filters("active_plugins", get_option("active_plugins")))) {
    require RVVUP_PLUGIN_PATH . "autoload.php";
    require RVVUP_PLUGIN_PATH . "boot.php";

    add_action("woocommerce_blocks_loaded", "rvvup_woocommerce_loaded");

    add_filter("plugin_action_links_" . plugin_basename(__FILE__), "rvvup_add_plugin_actions_links");

    /**
     * Load Gateway and Filters/Action after the woocommerce plugin is loaded,
     * if the Rvvup Plugin is not configured, then does nothing.
     */
    function rvvup_woocommerce_loaded()
    {
        if ((new Rvvup())->needs_setup()) {
            add_filter("woocommerce_payment_gateways", [GatewayLoader::class, "loadInitialSetupGateway"]);
            return;
        }
        //Load payment gateways
        new GatewayLoader();
        if (class_exists("Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType")) {
            add_action("woocommerce_blocks_payment_method_type_registration", function (
                Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry
            ) {
                $payment_method_registry->register(new Rvvup_WC_Blocks_Support());
            });
        }
        // Register Custom Post Types
        PostTypesManager::getInstance()->init();

        // Register Rvvup Refunds associated Services
        RvvupRefundRepository::getInstance()->init();
        RvvupRefundManager::getInstance()->init();
        RefundManager::getInstance()->init();

        // Register custom order statuses.
        OrderStatusManager::getInstance()->init();

        //Woocommerce filters and actions
        add_filter("woocommerce_short_description", "rvvup_addRestrictionPdpMessage");
        add_action("woocommerce_before_checkout_form", "rvvup_addRestrictionCheckoutMessage");

        ViewManager::getInstance()->init();
        AdminViewManager::getInstance()->init();

        add_action("woocommerce_product_options_general_product_data", "rvvup_addRestrictionField");
        add_action("woocommerce_process_product_meta", "rvvup_saveRestrictionField");

        /**
         * Load static asset files (css/js)
         */
        add_action("wp_enqueue_scripts", [AssetLoader::class, "load"]);
        add_action("admin_enqueue_scripts", [AssetLoader::class, "admin"]);
        register_deactivation_hook(RVVUP_PLUGIN_FILE, "rvvup_deactivatePlugin");
        register_uninstall_hook(RVVUP_PLUGIN_FILE, "rvvup_uninstallPlugin");

        /**
         * Register return and webhook URL with WooCommerce API
         */
        add_action("woocommerce_api_rvvup-webhook", [WebhookEndpoint::class, "execute"]);
        add_action("woocommerce_api_rvvup/webhook", [WebhookEndpoint::class, "execute"]);
        add_action("wp_ajax_rvvup_void", [VoidEndpoint::class, "execute"]);
        add_action("woocommerce_api_rvvup-return", [ReturnEndpoint::class, "execute"]);
        add_action("woocommerce_api_rvvup/return", [ReturnEndpoint::class, "execute"]);
        add_action("wc_ajax_" . PayOrderEndpoint::ENDPOINT, [PayOrderEndpoint::class, "execute"]);
        add_action("wc_ajax_" . ExpressEndpoint::ENDPOINT, [ExpressEndpoint::class, "execute"]);
        add_action("wc_ajax_" . ExpressUpdateEndpoint::ENDPOINT, [ExpressUpdateEndpoint::class, "execute"]);
        add_action("wc_ajax_" . ExpressDeleteEndpoint::ENDPOINT, [ExpressDeleteEndpoint::class, "execute"]);
        add_action("wc_ajax_" . CardSubmitCallbackEndpoint::ENDPOINT, [CardSubmitCallbackEndpoint::class, "execute"]);
        add_filter("woocommerce_checkout_get_value", [CheckoutPopulate::class, "rvvup_populate_checkout"], null, 2);

        /**
         * Ensure only Express payments show on Express Payments transactions
         */
        new ExpressCheckoutLoad();
        new ExpressCheckoutCartModification();
    }

    /**
     * Add a settings link to the plugin on the admin page
     */
    function rvvup_add_plugin_actions_links($links)
    {
        $settings_link = [
            '<a href="admin.php?page=wc-settings&tab=checkout&section=rvvup_gateway">' .
            __("Settings", "rvvup-for-woocommerce") .
            "</a>",
        ];
        return array_merge($settings_link, $links);
    }
    /**
     * Adds the checkbox to the product edit form in the admin
     */
    function rvvup_addRestrictionField()
    {
        woocommerce_wp_checkbox([
            "id" => "_rvvup_restricted",
            "label" => __("Is product restricted", "rvvup-for-woocommerce"),
        ]);
    }

    /**
     * Save the value of the checkbox to the product metadata when the product is saved
     * @param $post_id
     */
    function rvvup_saveRestrictionField($post_id)
    {
        $isRestricted = isset($_POST["_rvvup_restricted"]) ? "yes" : "no";
        if (class_exists(Automattic\WooCommerce\Utilities\OrderUtil::class)) {
            if (Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
                $product = wc_get_product($post_id);
                if ($product) {
                    $product->update_meta_data("_rvvup_restricted", esc_attr($isRestricted));
                    $product->save();
                }
            } else {
                update_post_meta($post_id, "_rvvup_restricted", esc_attr($isRestricted));
            }
        } else {
            update_post_meta($post_id, "_rvvup_restricted", esc_attr($isRestricted));
        }
    }

    /**
     * Add the restriction message to the product page if the item is restricted
     * @param $html
     * @return string
     */
    function rvvup_addRestrictionPdpMessage($html)
    {
        $id = get_the_ID();
        if (isset($id)) {
            $isRestricted = get_post_meta($id, "_rvvup_restricted", true);
            if ($isRestricted === "yes") {
                $rvvup = new Rvvup();
                $message = $rvvup->settings["restriction_message_pdp"];
                $html .= "<aside aria-live role='status'>$message</aside>";
            }
        }
        return $html;
    }

    /**
     * Show a notice message in the checkout if anything in the cart is restricted
     */
    function rvvup_addRestrictionCheckoutMessage()
    {
        $rvvup = new Rvvup();
        if ($rvvup->settings["restriction_message_checkout_enabled"] ?? "" === "yes") {
            foreach (WC()->cart->get_cart() as $item) {
                $id = $item["data"]->get_id();
                $isRestricted = get_post_meta($id, "_rvvup_restricted", true);
                if ($isRestricted === "yes") {
                    wc_add_notice($rvvup->settings["restriction_message_checkout"], "notice");
                    break;
                }
            }
        }
    }

    /**
     * @param string $method
     * @param string $url
     * @return string
     */
    function getCheckoutIconPath(string $method, string $url): string
    {
        if ($method === "CLEARPAY") {
            $theme = (new Rvvup())->settings["clearpay_theme"] ?? "black-on-mint";
            $path = sprintf("/assets/svg/%s/checkout/%s.svg", strtolower($method), $theme);
            if (file_exists(RVVUP_PLUGIN_PATH . $path)) {
                return RVVUP_PLUGIN_URL . $path . "?version=" . urlencode(Versions::getPluginVersion());
            }
        }

        return $url;
    }

    /**
     * Actions to perform when plugin is deactivated.
     * Executed only if WooCommerce is installed.
     *
     * @return void
     */
    function rvvup_deactivatePlugin()
    {
        if (!is_admin()) {
            return;
        }

        EventManager::getInstance()->dispatchPluginDisabledEvent("Rvvup Plugin Deactivated");
    }

    /**
     * Actions to perform when plugin is activated.
     *
     * @return void
     */
    function rvvup_activatePlugin()
    {
        if (!is_admin()) {
            return;
        }

        EventManager::getInstance()->dispatchPluginEnabledEvent("Rvvup Plugin Activated");
    }

    /**
     * Actions to perform when plugin is uninstalled.
     * Executed only if WooCommerce is installed.
     *
     * @return void
     */
    function rvvup_uninstallPlugin()
    {
        if (!is_admin()) {
            return;
        }

        EventManager::getInstance()->dispatchPluginDisabledEvent("Rvvup Plugin Uninstalled");
    }

    add_action("wp_ajax_rvvup_log", [LogController::class, "execute"]);

    /**
     * Supply order URL if present in order object
     */
    add_filter(
        "woocommerce_get_transaction_url",
        function ($url, $order, $method) {
            /** \WC_Order $order */
            if ($method instanceof Rvvup) {
                return $order->get_meta("_rvvup_url");
            }
            return $url;
        },
        10,
        3
    );

    /**
     * By default, the WooCommerce checkout does not output notices, so we will print them above the thank-you template
     */
    add_action(
        "woocommerce_before_template_part",
        function ($template) {
            if ("checkout/thankyou.php" === $template) {
                wc_print_notices();
            }
        },
        10,
        1
    );
    function add_void_button_to_order_edit_page($order)
    {
        $id = $order->get_id();
        $order = wc_get_order($id);
        if (VoidOrder::isVoidable($order)) {
            echo "<button class='button button-primary rvvup-void-button' id='$id'>Void</button>";
        }
    }

    add_action("woocommerce_admin_order_data_after_order_details", "add_void_button_to_order_edit_page");

    /**
     * Initialise the auto-update logic for the closed-beta stage
     */
    new ClosedBetaUpdater();
}

register_activation_hook(RVVUP_PLUGIN_PATH . "/" . RVVUP_PLUGIN_FILE, "rvvup_activatePlugin");
